/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.printqueueitem;

import java.net.URI;
import java.util.Map;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpUriRequest;

import com.google.common.collect.Maps;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.sap.cloud.sdk.cloudplatform.connectivity.HttpDestinationProperties;
import com.sap.cloud.sdk.datamodel.odata.helper.SingleValuedFluentHelperFunction;

/**
 * Fluent helper for the <b>Set_PrintParameter</b> OData function import.
 *
 */
public class Set_PrintParameterFluentHelper
    extends
    SingleValuedFluentHelperFunction<Set_PrintParameterFluentHelper, FuncImpReturn, FuncImpReturn>
{

    private final Map<String, Object> values = Maps.newHashMap();

    /**
     * Creates a fluent helper object that will execute the <b>Set_PrintParameter</b> OData function import with the
     * provided parameters. To perform execution, call the {@link #executeRequest executeRequest} method on the fluent
     * helper object.
     *
     * @param servicePath
     *            Service path to be used to call the functions against.
     * @param qname
     *            Constraints: Not nullable, Maximum length: 32
     *            <p>
     *            Original parameter name from the Odata EDM: <b>Qname</b>
     *            </p>
     * @param printParams
     *            Constraints: none
     *            <p>
     *            Original parameter name from the Odata EDM: <b>PrintParams</b>
     *            </p>
     */
    public Set_PrintParameterFluentHelper(
        @Nonnull final String servicePath,
        @Nonnull final String printParams,
        @Nonnull final String qname )
    {
        super(servicePath);
        values.put("PrintParams", printParams);
        values.put("Qname", qname);
    }

    @Override
    @Nonnull
    protected Class<FuncImpReturn> getEntityClass()
    {
        return FuncImpReturn.class;
    }

    @Override
    @Nonnull
    protected String getFunctionName()
    {
        return "Set_PrintParameter";
    }

    @Override
    @Nullable
    protected JsonElement refineJsonResponse( @Nullable JsonElement jsonElement )
    {
        if( (jsonElement instanceof JsonObject) && ((JsonObject) jsonElement).has(getFunctionName()) ) {
            jsonElement = ((JsonObject) jsonElement).get(getFunctionName());
        }
        return super.refineJsonResponse(jsonElement);
    }

    @Override
    @Nonnull
    protected Map<String, Object> getParameters()
    {
        return values;
    }

    @Override
    @Nonnull
    protected HttpUriRequest createRequest( @Nonnull final URI uri )
    {
        return new HttpGet(uri);
    }

    /**
     * Execute this function import.
     *
     */
    @Override
    @Nullable
    public FuncImpReturn executeRequest( @Nonnull final HttpDestinationProperties destination )
    {
        return super.executeSingle(destination);
    }

}
