/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.processorder;

import java.util.Map;

import javax.annotation.Nonnull;

import com.google.common.collect.Maps;
import com.sap.cloud.sdk.datamodel.odata.helper.FluentHelperByKey;
import com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.processorder.selectable.ProcessOrderItemSelectable;

/**
 * Fluent helper to fetch a single
 * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.processorder.ProcessOrderItem ProcessOrderItem} entity
 * using key fields. This fluent helper allows methods which modify the underlying query to be called before executing
 * the query itself.
 *
 */
public class ProcessOrderItemByKeyFluentHelper
    extends
    FluentHelperByKey<ProcessOrderItemByKeyFluentHelper, ProcessOrderItem, ProcessOrderItemSelectable>
{

    private final Map<String, Object> key = Maps.newHashMap();

    /**
     * Creates a fluent helper object that will fetch a single
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.processorder.ProcessOrderItem ProcessOrderItem} entity
     * with the provided key field values. To perform execution, call the {@link #executeRequest executeRequest} method
     * on the fluent helper object.
     *
     * @param entityCollection
     *            Entity Collection to be used to fetch a single {@code ProcessOrderItem}
     * @param servicePath
     *            Service path to be used to fetch a single {@code ProcessOrderItem}
     * @param manufacturingOrderItem
     *            Manufacturing Order Item ID
     *            <p>
     *            Constraints: Not nullable, Maximum length: 4
     *            </p>
     * @param manufacturingOrder
     *            Manufacturing Order ID
     *            <p>
     *            Constraints: Not nullable, Maximum length: 12
     *            </p>
     */
    public ProcessOrderItemByKeyFluentHelper(
        @Nonnull final String servicePath,
        @Nonnull final String entityCollection,
        final String manufacturingOrder,
        final String manufacturingOrderItem )
    {
        super(servicePath, entityCollection);
        this.key.put("ManufacturingOrder", manufacturingOrder);
        this.key.put("ManufacturingOrderItem", manufacturingOrderItem);
    }

    @Override
    @Nonnull
    protected Class<ProcessOrderItem> getEntityClass()
    {
        return ProcessOrderItem.class;
    }

    @Override
    @Nonnull
    protected Map<String, Object> getKey()
    {
        return key;
    }

}
