/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.productionorderconfirmationv2;

import java.net.URI;
import java.time.LocalDateTime;
import java.util.Map;
import java.util.UUID;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpUriRequest;

import com.google.common.collect.Maps;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.sap.cloud.sdk.cloudplatform.connectivity.HttpDestinationProperties;
import com.sap.cloud.sdk.datamodel.odata.helper.SingleValuedFluentHelperFunction;

/**
 * Fluent helper for the <b>CancelProdnOrdConf</b> OData function import.
 *
 */
public class CancelProdnOrdConfFluentHelper
    extends
    SingleValuedFluentHelperFunction<CancelProdnOrdConfFluentHelper, ProdnOrdConf2, ProdnOrdConf2>
{

    private final Map<String, Object> values = Maps.newHashMap();

    /**
     * Creates a fluent helper object that will execute the <b>CancelProdnOrdConf</b> OData function import with the
     * provided parameters. To perform execution, call the {@link #executeRequest executeRequest} method on the fluent
     * helper object.
     *
     * @param externalSystemConfirmation
     *            External Key
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Nullable
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>ExternalSystemConfirmation</b>
     *            </p>
     * @param confirmationCount
     *            Order Confirmation
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Not nullable, Maximum length: 8
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>ConfirmationCount</b>
     *            </p>
     * @param servicePath
     *            Service path to be used to call the functions against.
     * @param confirmationText
     *            Confirmation Text
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Nullable, Maximum length: 40
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>ConfirmationText</b>
     *            </p>
     * @param postingDate
     *            Posting Date
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Nullable, Precision: 0
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>PostingDate</b>
     *            </p>
     * @param confirmationGroup
     *            Confirmation Group
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Not nullable, Maximum length: 10
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>ConfirmationGroup</b>
     *            </p>
     */
    public CancelProdnOrdConfFluentHelper(
        @Nonnull final String servicePath,
        @Nonnull final String confirmationGroup,
        @Nonnull final String confirmationCount,
        @Nullable final LocalDateTime postingDate,
        @Nullable final String confirmationText,
        @Nullable final UUID externalSystemConfirmation )
    {
        super(servicePath);
        values.put("ConfirmationGroup", confirmationGroup);
        values.put("ConfirmationCount", confirmationCount);
        if( postingDate != null ) {
            values.put("PostingDate", postingDate);
        }
        if( confirmationText != null ) {
            values.put("ConfirmationText", confirmationText);
        }
        if( externalSystemConfirmation != null ) {
            values.put("ExternalSystemConfirmation", externalSystemConfirmation);
        }
    }

    @Override
    @Nonnull
    protected Class<ProdnOrdConf2> getEntityClass()
    {
        return ProdnOrdConf2.class;
    }

    @Override
    @Nonnull
    protected String getFunctionName()
    {
        return "CancelProdnOrdConf";
    }

    @Override
    @Nullable
    protected JsonElement refineJsonResponse( @Nullable JsonElement jsonElement )
    {
        if( (jsonElement instanceof JsonObject) && ((JsonObject) jsonElement).has(getFunctionName()) ) {
            jsonElement = ((JsonObject) jsonElement).get(getFunctionName());
        }
        return super.refineJsonResponse(jsonElement);
    }

    @Override
    @Nonnull
    protected Map<String, Object> getParameters()
    {
        return values;
    }

    @Override
    @Nonnull
    protected HttpUriRequest createRequest( @Nonnull final URI uri )
    {
        return new HttpPost(uri);
    }

    /**
     * Execute this function import.
     *
     */
    @Override
    @Nullable
    public ProdnOrdConf2 executeRequest( @Nonnull final HttpDestinationProperties destination )
    {
        return super.executeSingle(destination);
    }

}
