/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.productionvolumecapture;

import java.time.LocalDateTime;
import java.time.ZonedDateTime;
import java.util.Map;

import javax.annotation.Nonnull;

import com.google.common.collect.Maps;
import com.sap.cloud.sdk.datamodel.odata.helper.FluentHelperByKey;
import com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.productionvolumecapture.selectable.ProductionVolumeSelectable;

/**
 * Fluent helper to fetch a single
 * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.productionvolumecapture.ProductionVolume ProductionVolume}
 * entity using key fields. This fluent helper allows methods which modify the underlying query to be called before
 * executing the query itself.
 *
 */
public class ProductionVolumeByKeyFluentHelper
    extends
    FluentHelperByKey<ProductionVolumeByKeyFluentHelper, ProductionVolume, ProductionVolumeSelectable>
{

    private final Map<String, Object> key = Maps.newHashMap();

    /**
     * Creates a fluent helper object that will fetch a single
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.productionvolumecapture.ProductionVolume
     * ProductionVolume} entity with the provided key field values. To perform execution, call the
     * {@link #executeRequest executeRequest} method on the fluent helper object.
     *
     * @param entityCollection
     *            Entity Collection to be used to fetch a single {@code ProductionVolume}
     * @param productionDate
     *            Production Date
     *            <p>
     *            Constraints: Not nullable, Precision: 0
     *            </p>
     * @param servicePath
     *            Service path to be used to fetch a single {@code ProductionVolume}
     * @param well
     *            Well ID
     *            <p>
     *            Constraints: Not nullable, Maximum length: 30
     *            </p>
     * @param wellCompletion
     *            Well Completion ID
     *            <p>
     *            Constraints: Not nullable, Maximum length: 30
     *            </p>
     * @param oilAllocAtWellCompltnDateTime
     *            UTC Time Stamp in Short Form (YYYYMMDDhhmmss)
     *            <p>
     *            Constraints: Not nullable, Precision: 0
     *            </p>
     */
    public ProductionVolumeByKeyFluentHelper(
        @Nonnull final String servicePath,
        @Nonnull final String entityCollection,
        final String well,
        final String wellCompletion,
        final LocalDateTime productionDate,
        final ZonedDateTime oilAllocAtWellCompltnDateTime )
    {
        super(servicePath, entityCollection);
        this.key.put("Well", well);
        this.key.put("WellCompletion", wellCompletion);
        this.key.put("ProductionDate", productionDate);
        this.key.put("OilAllocAtWellCompltnDateTime", oilAllocAtWellCompltnDateTime);
    }

    @Override
    @Nonnull
    protected Class<ProductionVolume> getEntityClass()
    {
        return ProductionVolume.class;
    }

    @Override
    @Nonnull
    protected Map<String, Object> getKey()
    {
        return key;
    }

}
