/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.productmaster;

import java.util.Map;

import javax.annotation.Nonnull;

import com.google.common.collect.Maps;
import com.sap.cloud.sdk.datamodel.odata.helper.FluentHelperByKey;
import com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.productmaster.selectable.ProductMLPricesSelectable;

/**
 * Fluent helper to fetch a single
 * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.productmaster.ProductMLPrices ProductMLPrices} entity
 * using key fields. This fluent helper allows methods which modify the underlying query to be called before executing
 * the query itself.
 *
 */
public class ProductMLPricesByKeyFluentHelper
    extends
    FluentHelperByKey<ProductMLPricesByKeyFluentHelper, ProductMLPrices, ProductMLPricesSelectable>
{

    private final Map<String, Object> key = Maps.newHashMap();

    /**
     * Creates a fluent helper object that will fetch a single
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.productmaster.ProductMLPrices ProductMLPrices} entity
     * with the provided key field values. To perform execution, call the {@link #executeRequest executeRequest} method
     * on the fluent helper object.
     *
     * @param entityCollection
     *            Entity Collection to be used to fetch a single {@code ProductMLPrices}
     * @param product
     *            Material Number
     *            <p>
     *            Constraints: Not nullable, Maximum length: 40
     *            </p>
     * @param valuationArea
     *            Valuation Area
     *            <p>
     *            Constraints: Not nullable, Maximum length: 4
     *            </p>
     * @param servicePath
     *            Service path to be used to fetch a single {@code ProductMLPrices}
     * @param currencyRole
     *            External Currency Type and Valuation View
     *            <p>
     *            Constraints: Not nullable, Maximum length: 2
     *            </p>
     * @param valuationType
     *            Valuation Type
     *            <p>
     *            Constraints: Not nullable, Maximum length: 10
     *            </p>
     */
    public ProductMLPricesByKeyFluentHelper(
        @Nonnull final String servicePath,
        @Nonnull final String entityCollection,
        final String product,
        final String valuationArea,
        final String valuationType,
        final String currencyRole )
    {
        super(servicePath, entityCollection);
        this.key.put("Product", product);
        this.key.put("ValuationArea", valuationArea);
        this.key.put("ValuationType", valuationType);
        this.key.put("CurrencyRole", currencyRole);
    }

    @Override
    @Nonnull
    protected Class<ProductMLPrices> getEntityClass()
    {
        return ProductMLPrices.class;
    }

    @Override
    @Nonnull
    protected Map<String, Object> getKey()
    {
        return key;
    }

}
