/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.productmaster;

import java.util.Map;

import javax.annotation.Nonnull;

import com.google.common.collect.Maps;
import com.sap.cloud.sdk.datamodel.odata.helper.FluentHelperByKey;
import com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.productmaster.selectable.ProductUnitsOfMeasureSelectable;

/**
 * Fluent helper to fetch a single
 * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.productmaster.ProductUnitsOfMeasure ProductUnitsOfMeasure}
 * entity using key fields. This fluent helper allows methods which modify the underlying query to be called before
 * executing the query itself.
 *
 */
public class ProductUnitsOfMeasureByKeyFluentHelper
    extends
    FluentHelperByKey<ProductUnitsOfMeasureByKeyFluentHelper, ProductUnitsOfMeasure, ProductUnitsOfMeasureSelectable>
{

    private final Map<String, Object> key = Maps.newHashMap();

    /**
     * Creates a fluent helper object that will fetch a single
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.productmaster.ProductUnitsOfMeasure
     * ProductUnitsOfMeasure} entity with the provided key field values. To perform execution, call the
     * {@link #executeRequest executeRequest} method on the fluent helper object.
     *
     * @param entityCollection
     *            Entity Collection to be used to fetch a single {@code ProductUnitsOfMeasure}
     * @param product
     *            Product Number
     *            <p>
     *            Constraints: Not nullable, Maximum length: 40
     *            </p>
     * @param alternativeUnit
     *            Alternative Unit of Measure for Stockkeeping Unit
     *            <p>
     *            Constraints: Not nullable, Maximum length: 3
     *            </p>
     * @param servicePath
     *            Service path to be used to fetch a single {@code ProductUnitsOfMeasure}
     */
    public ProductUnitsOfMeasureByKeyFluentHelper(
        @Nonnull final String servicePath,
        @Nonnull final String entityCollection,
        final String product,
        final String alternativeUnit )
    {
        super(servicePath, entityCollection);
        this.key.put("Product", product);
        this.key.put("AlternativeUnit", alternativeUnit);
    }

    @Override
    @Nonnull
    protected Class<ProductUnitsOfMeasure> getEntityClass()
    {
        return ProductUnitsOfMeasure.class;
    }

    @Override
    @Nonnull
    protected Map<String, Object> getKey()
    {
        return key;
    }

}
