/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.productstructure;

import java.util.Map;
import java.util.UUID;

import javax.annotation.Nonnull;

import com.google.common.collect.Maps;
import com.sap.cloud.sdk.datamodel.odata.helper.FluentHelperByKey;
import com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.productstructure.selectable.ProductStructureVariantSelectable;

/**
 * Fluent helper to fetch a single
 * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.productstructure.ProductStructureVariant
 * ProductStructureVariant} entity using key fields. This fluent helper allows methods which modify the underlying query
 * to be called before executing the query itself.
 *
 */
public class ProductStructureVariantByKeyFluentHelper
    extends
    FluentHelperByKey<ProductStructureVariantByKeyFluentHelper, ProductStructureVariant, ProductStructureVariantSelectable>
{

    private final Map<String, Object> key = Maps.newHashMap();

    /**
     * Creates a fluent helper object that will fetch a single
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.productstructure.ProductStructureVariant
     * ProductStructureVariant} entity with the provided key field values. To perform execution, call the
     * {@link #executeRequest executeRequest} method on the fluent helper object.
     *
     * @param entityCollection
     *            Entity Collection to be used to fetch a single {@code ProductStructureVariant}
     * @param productStructureVariantUUID
     *            Internal Number of the PVS Variant
     *            <p>
     *            Constraints: Not nullable
     *            </p>
     * @param prodStrucVariantChgStateIntID
     *            Internal Counter for iPPE Objects
     *            <p>
     *            Constraints: Not nullable, Maximum length: 8
     *            </p>
     * @param servicePath
     *            Service path to be used to fetch a single {@code ProductStructureVariant}
     */
    public ProductStructureVariantByKeyFluentHelper(
        @Nonnull final String servicePath,
        @Nonnull final String entityCollection,
        final UUID productStructureVariantUUID,
        final String prodStrucVariantChgStateIntID )
    {
        super(servicePath, entityCollection);
        this.key.put("ProductStructureVariantUUID", productStructureVariantUUID);
        this.key.put("ProdStrucVariantChgStateIntID", prodStrucVariantChgStateIntID);
    }

    @Override
    @Nonnull
    protected Class<ProductStructureVariant> getEntityClass()
    {
        return ProductStructureVariant.class;
    }

    @Override
    @Nonnull
    protected Map<String, Object> getKey()
    {
        return key;
    }

}
