/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.purchasecontract;

import java.net.URI;
import java.util.List;
import java.util.Map;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpUriRequest;

import com.google.common.collect.Maps;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.sap.cloud.sdk.cloudplatform.connectivity.HttpDestinationProperties;
import com.sap.cloud.sdk.datamodel.odata.helper.CollectionValuedFluentHelperFunction;

/**
 * Fluent helper for the <b>ApproveDocument</b> OData function import.
 *
 */
public class ApproveDocumentFluentHelper
    extends
    CollectionValuedFluentHelperFunction<ApproveDocumentFluentHelper, Messages, List<Messages>>
{

    private final Map<String, Object> values = Maps.newHashMap();

    /**
     * Creates a fluent helper object that will execute the <b>ApproveDocument</b> OData function import with the
     * provided parameters. To perform execution, call the {@link #executeRequest executeRequest} method on the fluent
     * helper object.
     *
     * @param purchaseContract
     *            Constraints: Not nullable, Maximum length: 10
     *            <p>
     *            Original parameter name from the Odata EDM: <b>PurchaseContract</b>
     *            </p>
     * @param servicePath
     *            Service path to be used to call the functions against.
     */
    public ApproveDocumentFluentHelper( @Nonnull final String servicePath, @Nonnull final String purchaseContract )
    {
        super(servicePath);
        values.put("PurchaseContract", purchaseContract);
    }

    @Override
    @Nonnull
    protected Class<Messages> getEntityClass()
    {
        return Messages.class;
    }

    @Override
    @Nonnull
    protected String getFunctionName()
    {
        return "ApproveDocument";
    }

    @Override
    @Nullable
    protected JsonElement refineJsonResponse( @Nullable JsonElement jsonElement )
    {
        if( (jsonElement instanceof JsonObject) && ((JsonObject) jsonElement).has(getFunctionName()) ) {
            jsonElement = ((JsonObject) jsonElement).get(getFunctionName());
        }
        return super.refineJsonResponse(jsonElement);
    }

    @Override
    @Nonnull
    protected Map<String, Object> getParameters()
    {
        return values;
    }

    @Override
    @Nonnull
    protected HttpUriRequest createRequest( @Nonnull final URI uri )
    {
        return new HttpPost(uri);
    }

    /**
     * Execute this function import.
     *
     */
    @Override
    @Nonnull
    public List<Messages> executeRequest( @Nonnull final HttpDestinationProperties destination )
    {
        return super.executeMultiple(destination);
    }

}
