/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.purchasecontractv2;

import java.util.Map;

import javax.annotation.Nonnull;

import com.google.common.collect.Maps;
import com.sap.cloud.sdk.datamodel.odata.helper.FluentHelperByKey;
import com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.purchasecontractv2.selectable.PurCtrAddressSelectable;

/**
 * Fluent helper to fetch a single
 * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.purchasecontractv2.PurCtrAddress PurCtrAddress} entity
 * using key fields. This fluent helper allows methods which modify the underlying query to be called before executing
 * the query itself.
 *
 */
public class PurCtrAddressByKeyFluentHelper
    extends
    FluentHelperByKey<PurCtrAddressByKeyFluentHelper, PurCtrAddress, PurCtrAddressSelectable>
{

    private final Map<String, Object> key = Maps.newHashMap();

    /**
     * Creates a fluent helper object that will fetch a single
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.purchasecontractv2.PurCtrAddress PurCtrAddress} entity
     * with the provided key field values. To perform execution, call the {@link #executeRequest executeRequest} method
     * on the fluent helper object.
     *
     * @param entityCollection
     *            Entity Collection to be used to fetch a single {@code PurCtrAddress}
     * @param purchaseContract
     *            Purchasing Contract Header
     *            <p>
     *            Constraints: Not nullable, Maximum length: 10
     *            </p>
     * @param servicePath
     *            Service path to be used to fetch a single {@code PurCtrAddress}
     * @param purchaseContractItem
     *            Item Number of Purchasing Contract
     *            <p>
     *            Constraints: Not nullable, Maximum length: 5
     *            </p>
     * @param addressID
     *            Address Number
     *            <p>
     *            Constraints: Not nullable, Maximum length: 10
     *            </p>
     */
    public PurCtrAddressByKeyFluentHelper(
        @Nonnull final String servicePath,
        @Nonnull final String entityCollection,
        final String purchaseContract,
        final String addressID,
        final String purchaseContractItem )
    {
        super(servicePath, entityCollection);
        this.key.put("PurchaseContract", purchaseContract);
        this.key.put("AddressID", addressID);
        this.key.put("PurchaseContractItem", purchaseContractItem);
    }

    @Override
    @Nonnull
    protected Class<PurCtrAddress> getEntityClass()
    {
        return PurCtrAddress.class;
    }

    @Override
    @Nonnull
    protected Map<String, Object> getKey()
    {
        return key;
    }

}
