/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.purchaseorder;

import java.net.URI;
import java.util.Map;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpUriRequest;

import com.google.common.collect.Maps;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.sap.cloud.sdk.cloudplatform.connectivity.HttpDestinationProperties;
import com.sap.cloud.sdk.datamodel.odata.helper.SingleValuedFluentHelperFunction;

/**
 * Fluent helper for the <b>GetOutputBinaryData</b> OData function import.
 *
 */
public class GetOutputBinaryDataFluentHelper
    extends
    SingleValuedFluentHelperFunction<GetOutputBinaryDataFluentHelper, GetPDFResult, GetPDFResult>
{

    private final Map<String, Object> values = Maps.newHashMap();

    /**
     * Creates a fluent helper object that will execute the <b>GetOutputBinaryData</b> OData function import with the
     * provided parameters. To perform execution, call the {@link #executeRequest executeRequest} method on the fluent
     * helper object.
     *
     * @param servicePath
     *            Service path to be used to call the functions against.
     * @param purchaseOrder
     *            Purchase Order
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Nullable, Maximum length: 10
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>PurchaseOrder</b>
     *            </p>
     */
    public GetOutputBinaryDataFluentHelper( @Nonnull final String servicePath, @Nullable final String purchaseOrder )
    {
        super(servicePath);
        if( purchaseOrder != null ) {
            values.put("PurchaseOrder", purchaseOrder);
        }
    }

    @Override
    @Nonnull
    protected Class<GetPDFResult> getEntityClass()
    {
        return GetPDFResult.class;
    }

    @Override
    @Nonnull
    protected String getFunctionName()
    {
        return "GetOutputBinaryData";
    }

    @Override
    @Nullable
    protected JsonElement refineJsonResponse( @Nullable JsonElement jsonElement )
    {
        if( (jsonElement instanceof JsonObject) && ((JsonObject) jsonElement).has(getFunctionName()) ) {
            jsonElement = ((JsonObject) jsonElement).get(getFunctionName());
        }
        return super.refineJsonResponse(jsonElement);
    }

    @Override
    @Nonnull
    protected Map<String, Object> getParameters()
    {
        return values;
    }

    @Override
    @Nonnull
    protected HttpUriRequest createRequest( @Nonnull final URI uri )
    {
        return new HttpGet(uri);
    }

    /**
     * Execute this function import.
     *
     */
    @Override
    @Nullable
    public GetPDFResult executeRequest( @Nonnull final HttpDestinationProperties destination )
    {
        return super.executeSingle(destination);
    }

}
