/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.qualityinspectiondata;

import java.util.Map;

import javax.annotation.Nonnull;

import com.google.common.collect.Maps;
import com.sap.cloud.sdk.datamodel.odata.helper.FluentHelperByKey;
import com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.qualityinspectiondata.selectable.InspectionResultValueSelectable;

/**
 * Fluent helper to fetch a single
 * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.qualityinspectiondata.InspectionResultValue
 * InspectionResultValue} entity using key fields. This fluent helper allows methods which modify the underlying query
 * to be called before executing the query itself.
 *
 */
public class InspectionResultValueByKeyFluentHelper
    extends
    FluentHelperByKey<InspectionResultValueByKeyFluentHelper, InspectionResultValue, InspectionResultValueSelectable>
{

    private final Map<String, Object> key = Maps.newHashMap();

    /**
     * Creates a fluent helper object that will fetch a single
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.qualityinspectiondata.InspectionResultValue
     * InspectionResultValue} entity with the provided key field values. To perform execution, call the
     * {@link #executeRequest executeRequest} method on the fluent helper object.
     *
     * @param entityCollection
     *            Entity Collection to be used to fetch a single {@code InspectionResultValue}
     * @param servicePath
     *            Service path to be used to fetch a single {@code InspectionResultValue}
     * @param inspPlanOperationInternalID
     *            Current Node Number from Order Counter
     *            <p>
     *            Constraints: Not nullable, Maximum length: 8
     *            </p>
     * @param inspectionCharacteristic
     *            Inspection Characteristic Number
     *            <p>
     *            Constraints: Not nullable, Maximum length: 4
     *            </p>
     * @param inspResultValueInternalID
     *            Number of Individual Result
     *            <p>
     *            Constraints: Not nullable, Maximum length: 8
     *            </p>
     * @param inspectionLot
     *            Inspection Lot Number
     *            <p>
     *            Constraints: Not nullable, Maximum length: 12
     *            </p>
     */
    public InspectionResultValueByKeyFluentHelper(
        @Nonnull final String servicePath,
        @Nonnull final String entityCollection,
        final String inspectionLot,
        final String inspPlanOperationInternalID,
        final String inspectionCharacteristic,
        final String inspResultValueInternalID )
    {
        super(servicePath, entityCollection);
        this.key.put("InspectionLot", inspectionLot);
        this.key.put("InspPlanOperationInternalID", inspPlanOperationInternalID);
        this.key.put("InspectionCharacteristic", inspectionCharacteristic);
        this.key.put("InspResultValueInternalID", inspResultValueInternalID);
    }

    @Override
    @Nonnull
    protected Class<InspectionResultValue> getEntityClass()
    {
        return InspectionResultValue.class;
    }

    @Override
    @Nonnull
    protected Map<String, Object> getKey()
    {
        return key;
    }

}
