/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.recipeheaderandformula;

import java.util.Map;
import java.util.UUID;

import javax.annotation.Nonnull;

import com.google.common.collect.Maps;
import com.sap.cloud.sdk.datamodel.odata.helper.FluentHelperByKey;
import com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.recipeheaderandformula.selectable.RecipeCharcValueSelectable;

/**
 * Fluent helper to fetch a single
 * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.recipeheaderandformula.RecipeCharcValue RecipeCharcValue}
 * entity using key fields. This fluent helper allows methods which modify the underlying query to be called before
 * executing the query itself.
 *
 */
public class RecipeCharcValueByKeyFluentHelper
    extends
    FluentHelperByKey<RecipeCharcValueByKeyFluentHelper, RecipeCharcValue, RecipeCharcValueSelectable>
{

    private final Map<String, Object> key = Maps.newHashMap();

    /**
     * Creates a fluent helper object that will fetch a single
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.recipeheaderandformula.RecipeCharcValue
     * RecipeCharcValue} entity with the provided key field values. To perform execution, call the
     * {@link #executeRequest executeRequest} method on the fluent helper object.
     *
     * @param entityCollection
     *            Entity Collection to be used to fetch a single {@code RecipeCharcValue}
     * @param charcValuePositionNumber
     *            Characteristic value counter
     *            <p>
     *            Constraints: Not nullable, Maximum length: 3
     *            </p>
     * @param charcInternalID
     *            Characteristic Internal ID
     *            <p>
     *            Constraints: Not nullable, Maximum length: 10
     *            </p>
     * @param recipeUUID
     *            Common Recipe GUID
     *            <p>
     *            Constraints: Not nullable
     *            </p>
     * @param servicePath
     *            Service path to be used to fetch a single {@code RecipeCharcValue}
     * @param classType
     *            Class Type
     *            <p>
     *            Constraints: Not nullable, Maximum length: 3
     *            </p>
     */
    public RecipeCharcValueByKeyFluentHelper(
        @Nonnull final String servicePath,
        @Nonnull final String entityCollection,
        final UUID recipeUUID,
        final String classType,
        final String charcInternalID,
        final String charcValuePositionNumber )
    {
        super(servicePath, entityCollection);
        this.key.put("RecipeUUID", recipeUUID);
        this.key.put("ClassType", classType);
        this.key.put("CharcInternalID", charcInternalID);
        this.key.put("CharcValuePositionNumber", charcValuePositionNumber);
    }

    @Override
    @Nonnull
    protected Class<RecipeCharcValue> getEntityClass()
    {
        return RecipeCharcValue.class;
    }

    @Override
    @Nonnull
    protected Map<String, Object> getKey()
    {
        return key;
    }

}
