/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.recipeheaderandformula;

import java.util.Map;
import java.util.UUID;

import javax.annotation.Nonnull;

import com.google.common.collect.Maps;
import com.sap.cloud.sdk.datamodel.odata.helper.FluentHelperByKey;
import com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.recipeheaderandformula.selectable.RecipeClassSelectable;

/**
 * Fluent helper to fetch a single
 * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.recipeheaderandformula.RecipeClass RecipeClass} entity
 * using key fields. This fluent helper allows methods which modify the underlying query to be called before executing
 * the query itself.
 *
 */
public class RecipeClassByKeyFluentHelper
    extends
    FluentHelperByKey<RecipeClassByKeyFluentHelper, RecipeClass, RecipeClassSelectable>
{

    private final Map<String, Object> key = Maps.newHashMap();

    /**
     * Creates a fluent helper object that will fetch a single
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.recipeheaderandformula.RecipeClass RecipeClass} entity
     * with the provided key field values. To perform execution, call the {@link #executeRequest executeRequest} method
     * on the fluent helper object.
     *
     * @param entityCollection
     *            Entity Collection to be used to fetch a single {@code RecipeClass}
     * @param recipeUUID
     *            Common Recipe GUID
     *            <p>
     *            Constraints: Not nullable
     *            </p>
     * @param classInternalID
     *            Internal Class Number
     *            <p>
     *            Constraints: Not nullable, Maximum length: 10
     *            </p>
     * @param servicePath
     *            Service path to be used to fetch a single {@code RecipeClass}
     */
    public RecipeClassByKeyFluentHelper(
        @Nonnull final String servicePath,
        @Nonnull final String entityCollection,
        final UUID recipeUUID,
        final String classInternalID )
    {
        super(servicePath, entityCollection);
        this.key.put("RecipeUUID", recipeUUID);
        this.key.put("ClassInternalID", classInternalID);
    }

    @Override
    @Nonnull
    protected Class<RecipeClass> getEntityClass()
    {
        return RecipeClass.class;
    }

    @Override
    @Nonnull
    protected Map<String, Object> getKey()
    {
        return key;
    }

}
