/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.reservationdocument;

import java.util.Map;

import javax.annotation.Nonnull;

import com.google.common.collect.Maps;
import com.sap.cloud.sdk.datamodel.odata.helper.FluentHelperByKey;
import com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.reservationdocument.selectable.ReservationDocumentItemSelectable;

/**
 * Fluent helper to fetch a single
 * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.reservationdocument.ReservationDocumentItem
 * ReservationDocumentItem} entity using key fields. This fluent helper allows methods which modify the underlying query
 * to be called before executing the query itself.
 *
 */
public class ReservationDocumentItemByKeyFluentHelper
    extends
    FluentHelperByKey<ReservationDocumentItemByKeyFluentHelper, ReservationDocumentItem, ReservationDocumentItemSelectable>
{

    private final Map<String, Object> key = Maps.newHashMap();

    /**
     * Creates a fluent helper object that will fetch a single
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.reservationdocument.ReservationDocumentItem
     * ReservationDocumentItem} entity with the provided key field values. To perform execution, call the
     * {@link #executeRequest executeRequest} method on the fluent helper object.
     *
     * @param entityCollection
     *            Entity Collection to be used to fetch a single {@code ReservationDocumentItem}
     * @param reservationItem
     *            Item Number of Reservation / Dependent Requirements
     *            <p>
     *            Constraints: Not nullable, Maximum length: 4
     *            </p>
     * @param recordType
     *            Record type
     *            <p>
     *            Constraints: Not nullable, Maximum length: 1
     *            </p>
     * @param servicePath
     *            Service path to be used to fetch a single {@code ReservationDocumentItem}
     * @param reservation
     *            Number of reservation/dependent requirements
     *            <p>
     *            Constraints: Not nullable, Maximum length: 10
     *            </p>
     */
    public ReservationDocumentItemByKeyFluentHelper(
        @Nonnull final String servicePath,
        @Nonnull final String entityCollection,
        final String reservation,
        final String reservationItem,
        final String recordType )
    {
        super(servicePath, entityCollection);
        this.key.put("Reservation", reservation);
        this.key.put("ReservationItem", reservationItem);
        this.key.put("RecordType", recordType);
    }

    @Override
    @Nonnull
    protected Class<ReservationDocumentItem> getEntityClass()
    {
        return ReservationDocumentItem.class;
    }

    @Override
    @Nonnull
    protected Map<String, Object> getKey()
    {
        return key;
    }

}
