/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.salescontract;

import java.math.BigDecimal;
import java.time.LocalDateTime;
import java.util.List;
import java.util.Map;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.databind.annotation.JsonDeserialize;
import com.fasterxml.jackson.databind.annotation.JsonSerialize;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.gson.annotations.JsonAdapter;
import com.google.gson.annotations.SerializedName;
import com.sap.cloud.sdk.cloudplatform.connectivity.HttpDestinationProperties;
import com.sap.cloud.sdk.datamodel.odata.client.exception.ODataException;
import com.sap.cloud.sdk.datamodel.odata.helper.VdmEntity;
import com.sap.cloud.sdk.s4hana.datamodel.odata.adapter.ODataField;
import com.sap.cloud.sdk.s4hana.datamodel.odata.annotation.Key;
import com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.salescontract.field.SalesContractItemField;
import com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.salescontract.link.SalesContractItemLink;
import com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.salescontract.link.SalesContractItemOneToOneLink;
import com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.salescontract.selectable.SalesContractItemSelectable;
import com.sap.cloud.sdk.typeconverter.TypeConverter;

import io.vavr.control.Option;
import lombok.AccessLevel;
import lombok.AllArgsConstructor;
import lombok.Builder;
import lombok.Data;
import lombok.EqualsAndHashCode;
import lombok.Getter;
import lombok.NoArgsConstructor;
import lombok.Setter;
import lombok.ToString;

/**
 * Sales Contract Item
 * <p>
 * </p>
 * <p>
 * Original entity name from the Odata EDM: <b>A_SalesContractItemType</b>
 * </p>
 *
 */
@Builder
@Data
@NoArgsConstructor
@AllArgsConstructor
@ToString( doNotUseGetters = true, callSuper = true )
@EqualsAndHashCode( doNotUseGetters = true, callSuper = true )
@JsonAdapter( com.sap.cloud.sdk.s4hana.datamodel.odata.adapter.ODataVdmEntityAdapterFactory.class )
public class SalesContractItem extends VdmEntity<SalesContractItem>
{

    /**
     * Selector for all available fields of SalesContractItem.
     *
     */
    public final static SalesContractItemSelectable ALL_FIELDS = () -> "*";
    /**
     * (Key Field) Constraints: Not nullable, Maximum length: 10
     * <p>
     * Original property name from the Odata EDM: <b>SalesContract</b>
     * </p>
     *
     * @return Sales Contract
     */
    @Key
    @SerializedName( "SalesContract" )
    @JsonProperty( "SalesContract" )
    @Nullable
    @ODataField( odataName = "SalesContract" )
    private String salesContract;
    /**
     * Use with available fluent helpers to apply the <b>SalesContract</b> field to query operations.
     *
     */
    public final static SalesContractItemField<String> SALES_CONTRACT =
        new SalesContractItemField<String>("SalesContract");
    /**
     * (Key Field) Constraints: Not nullable, Maximum length: 6
     * <p>
     * Original property name from the Odata EDM: <b>SalesContractItem</b>
     * </p>
     *
     * @return Sales Contract Item
     */
    @Key
    @SerializedName( "SalesContractItem" )
    @JsonProperty( "SalesContractItem" )
    @Nullable
    @ODataField( odataName = "SalesContractItem" )
    private String salesContractItem;
    /**
     * Use with available fluent helpers to apply the <b>SalesContractItem</b> field to query operations.
     *
     */
    public final static SalesContractItemField<String> SALES_CONTRACT_ITEM =
        new SalesContractItemField<String>("SalesContractItem");
    /**
     * Constraints: Not nullable, Maximum length: 6
     * <p>
     * Original property name from the Odata EDM: <b>HigherLevelItem</b>
     * </p>
     *
     * @return Higher-Level Item in Bill of Material Structures
     */
    @SerializedName( "HigherLevelItem" )
    @JsonProperty( "HigherLevelItem" )
    @Nullable
    @ODataField( odataName = "HigherLevelItem" )
    private String higherLevelItem;
    /**
     * Use with available fluent helpers to apply the <b>HigherLevelItem</b> field to query operations.
     *
     */
    public final static SalesContractItemField<String> HIGHER_LEVEL_ITEM =
        new SalesContractItemField<String>("HigherLevelItem");
    /**
     * Constraints: Not nullable, Maximum length: 4
     * <p>
     * Original property name from the Odata EDM: <b>SalesContractItemCategory</b>
     * </p>
     *
     * @return Sales Document Item Category
     */
    @SerializedName( "SalesContractItemCategory" )
    @JsonProperty( "SalesContractItemCategory" )
    @Nullable
    @ODataField( odataName = "SalesContractItemCategory" )
    private String salesContractItemCategory;
    /**
     * Use with available fluent helpers to apply the <b>SalesContractItemCategory</b> field to query operations.
     *
     */
    public final static SalesContractItemField<String> SALES_CONTRACT_ITEM_CATEGORY =
        new SalesContractItemField<String>("SalesContractItemCategory");
    /**
     * Constraints: Not nullable, Maximum length: 40
     * <p>
     * Original property name from the Odata EDM: <b>SalesContractItemText</b>
     * </p>
     *
     * @return Short Text for Sales Order Item
     */
    @SerializedName( "SalesContractItemText" )
    @JsonProperty( "SalesContractItemText" )
    @Nullable
    @ODataField( odataName = "SalesContractItemText" )
    private String salesContractItemText;
    /**
     * Use with available fluent helpers to apply the <b>SalesContractItemText</b> field to query operations.
     *
     */
    public final static SalesContractItemField<String> SALES_CONTRACT_ITEM_TEXT =
        new SalesContractItemField<String>("SalesContractItemText");
    /**
     * Constraints: Not nullable, Maximum length: 35
     * <p>
     * Original property name from the Odata EDM: <b>PurchaseOrderByCustomer</b>
     * </p>
     *
     * @return Customer Reference
     */
    @SerializedName( "PurchaseOrderByCustomer" )
    @JsonProperty( "PurchaseOrderByCustomer" )
    @Nullable
    @ODataField( odataName = "PurchaseOrderByCustomer" )
    private String purchaseOrderByCustomer;
    /**
     * Use with available fluent helpers to apply the <b>PurchaseOrderByCustomer</b> field to query operations.
     *
     */
    public final static SalesContractItemField<String> PURCHASE_ORDER_BY_CUSTOMER =
        new SalesContractItemField<String>("PurchaseOrderByCustomer");
    /**
     * Constraints: Not nullable, Maximum length: 40
     * <p>
     * Original property name from the Odata EDM: <b>Material</b>
     * </p>
     *
     * @return Material Number
     */
    @SerializedName( "Material" )
    @JsonProperty( "Material" )
    @Nullable
    @ODataField( odataName = "Material" )
    private String material;
    /**
     * Use with available fluent helpers to apply the <b>Material</b> field to query operations.
     *
     */
    public final static SalesContractItemField<String> MATERIAL = new SalesContractItemField<String>("Material");
    /**
     * Constraints: Not nullable, Maximum length: 35
     * <p>
     * Original property name from the Odata EDM: <b>MaterialByCustomer</b>
     * </p>
     *
     * @return Material Number Used by Customer
     */
    @SerializedName( "MaterialByCustomer" )
    @JsonProperty( "MaterialByCustomer" )
    @Nullable
    @ODataField( odataName = "MaterialByCustomer" )
    private String materialByCustomer;
    /**
     * Use with available fluent helpers to apply the <b>MaterialByCustomer</b> field to query operations.
     *
     */
    public final static SalesContractItemField<String> MATERIAL_BY_CUSTOMER =
        new SalesContractItemField<String>("MaterialByCustomer");
    /**
     * Constraints: Not nullable, Precision: 0
     * <p>
     * Original property name from the Odata EDM: <b>PricingDate</b>
     * </p>
     *
     * @return Date for Pricing and Exchange Rate
     */
    @SerializedName( "PricingDate" )
    @JsonProperty( "PricingDate" )
    @Nullable
    @JsonSerialize( using = com.sap.cloud.sdk.s4hana.datamodel.odata.adapter.JacksonLocalDateTimeSerializer.class )
    @JsonDeserialize( using = com.sap.cloud.sdk.s4hana.datamodel.odata.adapter.JacksonLocalDateTimeDeserializer.class )
    @JsonAdapter( com.sap.cloud.sdk.s4hana.datamodel.odata.adapter.LocalDateTimeAdapter.class )
    @ODataField(
        odataName = "PricingDate",
        converter = com.sap.cloud.sdk.s4hana.datamodel.odata.adapter.LocalDateTimeCalendarConverter.class )
    private LocalDateTime pricingDate;
    /**
     * Use with available fluent helpers to apply the <b>PricingDate</b> field to query operations.
     *
     */
    public final static SalesContractItemField<LocalDateTime> PRICING_DATE =
        new SalesContractItemField<LocalDateTime>("PricingDate");
    /**
     * Constraints: Not nullable, Precision: 15, Scale: 3
     * <p>
     * Original property name from the Odata EDM: <b>RequestedQuantity</b>
     * </p>
     *
     * @return Requested Quantity
     */
    @SerializedName( "RequestedQuantity" )
    @JsonProperty( "RequestedQuantity" )
    @Nullable
    @ODataField( odataName = "RequestedQuantity" )
    private BigDecimal requestedQuantity;
    /**
     * Use with available fluent helpers to apply the <b>RequestedQuantity</b> field to query operations.
     *
     */
    public final static SalesContractItemField<BigDecimal> REQUESTED_QUANTITY =
        new SalesContractItemField<BigDecimal>("RequestedQuantity");
    /**
     * Constraints: Not nullable, Maximum length: 3
     * <p>
     * Original property name from the Odata EDM: <b>RequestedQuantityUnit</b>
     * </p>
     *
     * @return Unit of the Requested Quantity
     */
    @SerializedName( "RequestedQuantityUnit" )
    @JsonProperty( "RequestedQuantityUnit" )
    @Nullable
    @ODataField( odataName = "RequestedQuantityUnit" )
    private String requestedQuantityUnit;
    /**
     * Use with available fluent helpers to apply the <b>RequestedQuantityUnit</b> field to query operations.
     *
     */
    public final static SalesContractItemField<String> REQUESTED_QUANTITY_UNIT =
        new SalesContractItemField<String>("RequestedQuantityUnit");
    /**
     * Constraints: Not nullable, Maximum length: 3
     * <p>
     * Original property name from the Odata EDM: <b>RequestedQuantitySAPUnit</b>
     * </p>
     *
     * @return SAP Unit Code for Requested Quantity
     */
    @SerializedName( "RequestedQuantitySAPUnit" )
    @JsonProperty( "RequestedQuantitySAPUnit" )
    @Nullable
    @ODataField( odataName = "RequestedQuantitySAPUnit" )
    private String requestedQuantitySAPUnit;
    /**
     * Use with available fluent helpers to apply the <b>RequestedQuantitySAPUnit</b> field to query operations.
     *
     */
    public final static SalesContractItemField<String> REQUESTED_QUANTITY_SAP_UNIT =
        new SalesContractItemField<String>("RequestedQuantitySAPUnit");
    /**
     * Constraints: Not nullable, Maximum length: 3
     * <p>
     * Original property name from the Odata EDM: <b>RequestedQuantityISOUnit</b>
     * </p>
     *
     * @return ISO Unit Code for Requested Quantity
     */
    @SerializedName( "RequestedQuantityISOUnit" )
    @JsonProperty( "RequestedQuantityISOUnit" )
    @Nullable
    @ODataField( odataName = "RequestedQuantityISOUnit" )
    private String requestedQuantityISOUnit;
    /**
     * Use with available fluent helpers to apply the <b>RequestedQuantityISOUnit</b> field to query operations.
     *
     */
    public final static SalesContractItemField<String> REQUESTED_QUANTITY_ISO_UNIT =
        new SalesContractItemField<String>("RequestedQuantityISOUnit");
    /**
     * Constraints: Not nullable, Precision: 15, Scale: 3
     * <p>
     * Original property name from the Odata EDM: <b>ItemGrossWeight</b>
     * </p>
     *
     * @return Gross Weight of the Item
     */
    @SerializedName( "ItemGrossWeight" )
    @JsonProperty( "ItemGrossWeight" )
    @Nullable
    @ODataField( odataName = "ItemGrossWeight" )
    private BigDecimal itemGrossWeight;
    /**
     * Use with available fluent helpers to apply the <b>ItemGrossWeight</b> field to query operations.
     *
     */
    public final static SalesContractItemField<BigDecimal> ITEM_GROSS_WEIGHT =
        new SalesContractItemField<BigDecimal>("ItemGrossWeight");
    /**
     * Constraints: Not nullable, Precision: 15, Scale: 3
     * <p>
     * Original property name from the Odata EDM: <b>ItemNetWeight</b>
     * </p>
     *
     * @return Net Weight of the Item
     */
    @SerializedName( "ItemNetWeight" )
    @JsonProperty( "ItemNetWeight" )
    @Nullable
    @ODataField( odataName = "ItemNetWeight" )
    private BigDecimal itemNetWeight;
    /**
     * Use with available fluent helpers to apply the <b>ItemNetWeight</b> field to query operations.
     *
     */
    public final static SalesContractItemField<BigDecimal> ITEM_NET_WEIGHT =
        new SalesContractItemField<BigDecimal>("ItemNetWeight");
    /**
     * Constraints: Not nullable, Maximum length: 3
     * <p>
     * Original property name from the Odata EDM: <b>ItemWeightUnit</b>
     * </p>
     *
     * @return Unit of Weight
     */
    @SerializedName( "ItemWeightUnit" )
    @JsonProperty( "ItemWeightUnit" )
    @Nullable
    @ODataField( odataName = "ItemWeightUnit" )
    private String itemWeightUnit;
    /**
     * Use with available fluent helpers to apply the <b>ItemWeightUnit</b> field to query operations.
     *
     */
    public final static SalesContractItemField<String> ITEM_WEIGHT_UNIT =
        new SalesContractItemField<String>("ItemWeightUnit");
    /**
     * Constraints: Not nullable, Maximum length: 3
     * <p>
     * Original property name from the Odata EDM: <b>ItemWeightSAPUnit</b>
     * </p>
     *
     * @return SAP Unit Code for Item Weight
     */
    @SerializedName( "ItemWeightSAPUnit" )
    @JsonProperty( "ItemWeightSAPUnit" )
    @Nullable
    @ODataField( odataName = "ItemWeightSAPUnit" )
    private String itemWeightSAPUnit;
    /**
     * Use with available fluent helpers to apply the <b>ItemWeightSAPUnit</b> field to query operations.
     *
     */
    public final static SalesContractItemField<String> ITEM_WEIGHT_SAP_UNIT =
        new SalesContractItemField<String>("ItemWeightSAPUnit");
    /**
     * Constraints: Not nullable, Maximum length: 3
     * <p>
     * Original property name from the Odata EDM: <b>ItemWeightISOUnit</b>
     * </p>
     *
     * @return ISO Unit Code for Item Weight
     */
    @SerializedName( "ItemWeightISOUnit" )
    @JsonProperty( "ItemWeightISOUnit" )
    @Nullable
    @ODataField( odataName = "ItemWeightISOUnit" )
    private String itemWeightISOUnit;
    /**
     * Use with available fluent helpers to apply the <b>ItemWeightISOUnit</b> field to query operations.
     *
     */
    public final static SalesContractItemField<String> ITEM_WEIGHT_ISO_UNIT =
        new SalesContractItemField<String>("ItemWeightISOUnit");
    /**
     * Constraints: Not nullable, Precision: 15, Scale: 3
     * <p>
     * Original property name from the Odata EDM: <b>ItemVolume</b>
     * </p>
     *
     * @return Volume of the item
     */
    @SerializedName( "ItemVolume" )
    @JsonProperty( "ItemVolume" )
    @Nullable
    @ODataField( odataName = "ItemVolume" )
    private BigDecimal itemVolume;
    /**
     * Use with available fluent helpers to apply the <b>ItemVolume</b> field to query operations.
     *
     */
    public final static SalesContractItemField<BigDecimal> ITEM_VOLUME =
        new SalesContractItemField<BigDecimal>("ItemVolume");
    /**
     * Constraints: Not nullable, Maximum length: 3
     * <p>
     * Original property name from the Odata EDM: <b>ItemVolumeUnit</b>
     * </p>
     *
     * @return Volume Unit
     */
    @SerializedName( "ItemVolumeUnit" )
    @JsonProperty( "ItemVolumeUnit" )
    @Nullable
    @ODataField( odataName = "ItemVolumeUnit" )
    private String itemVolumeUnit;
    /**
     * Use with available fluent helpers to apply the <b>ItemVolumeUnit</b> field to query operations.
     *
     */
    public final static SalesContractItemField<String> ITEM_VOLUME_UNIT =
        new SalesContractItemField<String>("ItemVolumeUnit");
    /**
     * Constraints: Not nullable, Maximum length: 3
     * <p>
     * Original property name from the Odata EDM: <b>ItemVolumeSAPUnit</b>
     * </p>
     *
     * @return SAP Unit Code for Item Volume
     */
    @SerializedName( "ItemVolumeSAPUnit" )
    @JsonProperty( "ItemVolumeSAPUnit" )
    @Nullable
    @ODataField( odataName = "ItemVolumeSAPUnit" )
    private String itemVolumeSAPUnit;
    /**
     * Use with available fluent helpers to apply the <b>ItemVolumeSAPUnit</b> field to query operations.
     *
     */
    public final static SalesContractItemField<String> ITEM_VOLUME_SAP_UNIT =
        new SalesContractItemField<String>("ItemVolumeSAPUnit");
    /**
     * Constraints: Not nullable, Maximum length: 3
     * <p>
     * Original property name from the Odata EDM: <b>ItemVolumeISOUnit</b>
     * </p>
     *
     * @return ISO Unit Code for Item Volume
     */
    @SerializedName( "ItemVolumeISOUnit" )
    @JsonProperty( "ItemVolumeISOUnit" )
    @Nullable
    @ODataField( odataName = "ItemVolumeISOUnit" )
    private String itemVolumeISOUnit;
    /**
     * Use with available fluent helpers to apply the <b>ItemVolumeISOUnit</b> field to query operations.
     *
     */
    public final static SalesContractItemField<String> ITEM_VOLUME_ISO_UNIT =
        new SalesContractItemField<String>("ItemVolumeISOUnit");
    /**
     * Constraints: Not nullable, Precision: 14, Scale: 3
     * <p>
     * Original property name from the Odata EDM: <b>OutlineAgreementTargetAmount</b>
     * </p>
     *
     * @return Target Value for Outline Agreement in Document Currency
     */
    @SerializedName( "OutlineAgreementTargetAmount" )
    @JsonProperty( "OutlineAgreementTargetAmount" )
    @Nullable
    @ODataField( odataName = "OutlineAgreementTargetAmount" )
    private BigDecimal outlineAgreementTargetAmount;
    /**
     * Use with available fluent helpers to apply the <b>OutlineAgreementTargetAmount</b> field to query operations.
     *
     */
    public final static SalesContractItemField<BigDecimal> OUTLINE_AGREEMENT_TARGET_AMOUNT =
        new SalesContractItemField<BigDecimal>("OutlineAgreementTargetAmount");
    /**
     * Constraints: Not nullable, Maximum length: 5
     * <p>
     * Original property name from the Odata EDM: <b>TransactionCurrency</b>
     * </p>
     *
     * @return SD Document Currency
     */
    @SerializedName( "TransactionCurrency" )
    @JsonProperty( "TransactionCurrency" )
    @Nullable
    @ODataField( odataName = "TransactionCurrency" )
    private String transactionCurrency;
    /**
     * Use with available fluent helpers to apply the <b>TransactionCurrency</b> field to query operations.
     *
     */
    public final static SalesContractItemField<String> TRANSACTION_CURRENCY =
        new SalesContractItemField<String>("TransactionCurrency");
    /**
     * Constraints: Not nullable, Precision: 16, Scale: 3
     * <p>
     * Original property name from the Odata EDM: <b>NetAmount</b>
     * </p>
     *
     * @return Net Value of the Order Item in Document Currency
     */
    @SerializedName( "NetAmount" )
    @JsonProperty( "NetAmount" )
    @Nullable
    @ODataField( odataName = "NetAmount" )
    private BigDecimal netAmount;
    /**
     * Use with available fluent helpers to apply the <b>NetAmount</b> field to query operations.
     *
     */
    public final static SalesContractItemField<BigDecimal> NET_AMOUNT =
        new SalesContractItemField<BigDecimal>("NetAmount");
    /**
     * Constraints: Not nullable, Maximum length: 9
     * <p>
     * Original property name from the Odata EDM: <b>MaterialGroup</b>
     * </p>
     *
     * @return Material Group
     */
    @SerializedName( "MaterialGroup" )
    @JsonProperty( "MaterialGroup" )
    @Nullable
    @ODataField( odataName = "MaterialGroup" )
    private String materialGroup;
    /**
     * Use with available fluent helpers to apply the <b>MaterialGroup</b> field to query operations.
     *
     */
    public final static SalesContractItemField<String> MATERIAL_GROUP =
        new SalesContractItemField<String>("MaterialGroup");
    /**
     * Constraints: Not nullable, Maximum length: 2
     * <p>
     * Original property name from the Odata EDM: <b>MaterialPricingGroup</b>
     * </p>
     *
     * @return Material Price Group
     */
    @SerializedName( "MaterialPricingGroup" )
    @JsonProperty( "MaterialPricingGroup" )
    @Nullable
    @ODataField( odataName = "MaterialPricingGroup" )
    private String materialPricingGroup;
    /**
     * Use with available fluent helpers to apply the <b>MaterialPricingGroup</b> field to query operations.
     *
     */
    public final static SalesContractItemField<String> MATERIAL_PRICING_GROUP =
        new SalesContractItemField<String>("MaterialPricingGroup");
    /**
     * Constraints: Not nullable, Maximum length: 10
     * <p>
     * Original property name from the Odata EDM: <b>Batch</b>
     * </p>
     *
     * @return Batch Number
     */
    @SerializedName( "Batch" )
    @JsonProperty( "Batch" )
    @Nullable
    @ODataField( odataName = "Batch" )
    private String batch;
    /**
     * Use with available fluent helpers to apply the <b>Batch</b> field to query operations.
     *
     */
    public final static SalesContractItemField<String> BATCH = new SalesContractItemField<String>("Batch");
    /**
     * Constraints: Not nullable, Maximum length: 4
     * <p>
     * Original property name from the Odata EDM: <b>ProductionPlant</b>
     * </p>
     *
     * @return Plant (Own or External)
     */
    @SerializedName( "ProductionPlant" )
    @JsonProperty( "ProductionPlant" )
    @Nullable
    @ODataField( odataName = "ProductionPlant" )
    private String productionPlant;
    /**
     * Use with available fluent helpers to apply the <b>ProductionPlant</b> field to query operations.
     *
     */
    public final static SalesContractItemField<String> PRODUCTION_PLANT =
        new SalesContractItemField<String>("ProductionPlant");
    /**
     * Constraints: Not nullable, Maximum length: 4
     * <p>
     * Original property name from the Odata EDM: <b>StorageLocation</b>
     * </p>
     *
     * @return Storage Location
     */
    @SerializedName( "StorageLocation" )
    @JsonProperty( "StorageLocation" )
    @Nullable
    @ODataField( odataName = "StorageLocation" )
    private String storageLocation;
    /**
     * Use with available fluent helpers to apply the <b>StorageLocation</b> field to query operations.
     *
     */
    public final static SalesContractItemField<String> STORAGE_LOCATION =
        new SalesContractItemField<String>("StorageLocation");
    /**
     * Constraints: Not nullable, Maximum length: 4
     * <p>
     * Original property name from the Odata EDM: <b>ShippingPoint</b>
     * </p>
     *
     * @return Shipping Point / Receiving Point
     */
    @SerializedName( "ShippingPoint" )
    @JsonProperty( "ShippingPoint" )
    @Nullable
    @ODataField( odataName = "ShippingPoint" )
    private String shippingPoint;
    /**
     * Use with available fluent helpers to apply the <b>ShippingPoint</b> field to query operations.
     *
     */
    public final static SalesContractItemField<String> SHIPPING_POINT =
        new SalesContractItemField<String>("ShippingPoint");
    /**
     * Constraints: Not nullable, Maximum length: 3
     * <p>
     * Original property name from the Odata EDM: <b>IncotermsClassification</b>
     * </p>
     *
     * @return Incoterms (Part 1)
     */
    @SerializedName( "IncotermsClassification" )
    @JsonProperty( "IncotermsClassification" )
    @Nullable
    @ODataField( odataName = "IncotermsClassification" )
    private String incotermsClassification;
    /**
     * Use with available fluent helpers to apply the <b>IncotermsClassification</b> field to query operations.
     *
     */
    public final static SalesContractItemField<String> INCOTERMS_CLASSIFICATION =
        new SalesContractItemField<String>("IncotermsClassification");
    /**
     * Constraints: Not nullable, Maximum length: 28
     * <p>
     * Original property name from the Odata EDM: <b>IncotermsTransferLocation</b>
     * </p>
     *
     * @return Incoterms (Part 2)
     */
    @SerializedName( "IncotermsTransferLocation" )
    @JsonProperty( "IncotermsTransferLocation" )
    @Nullable
    @ODataField( odataName = "IncotermsTransferLocation" )
    private String incotermsTransferLocation;
    /**
     * Use with available fluent helpers to apply the <b>IncotermsTransferLocation</b> field to query operations.
     *
     */
    public final static SalesContractItemField<String> INCOTERMS_TRANSFER_LOCATION =
        new SalesContractItemField<String>("IncotermsTransferLocation");
    /**
     * Constraints: Not nullable, Maximum length: 70
     * <p>
     * Original property name from the Odata EDM: <b>IncotermsLocation1</b>
     * </p>
     *
     * @return Incoterms Location 1
     */
    @SerializedName( "IncotermsLocation1" )
    @JsonProperty( "IncotermsLocation1" )
    @Nullable
    @ODataField( odataName = "IncotermsLocation1" )
    private String incotermsLocation1;
    /**
     * Use with available fluent helpers to apply the <b>IncotermsLocation1</b> field to query operations.
     *
     */
    public final static SalesContractItemField<String> INCOTERMS_LOCATION1 =
        new SalesContractItemField<String>("IncotermsLocation1");
    /**
     * Constraints: Not nullable, Maximum length: 70
     * <p>
     * Original property name from the Odata EDM: <b>IncotermsLocation2</b>
     * </p>
     *
     * @return Incoterms Location 2
     */
    @SerializedName( "IncotermsLocation2" )
    @JsonProperty( "IncotermsLocation2" )
    @Nullable
    @ODataField( odataName = "IncotermsLocation2" )
    private String incotermsLocation2;
    /**
     * Use with available fluent helpers to apply the <b>IncotermsLocation2</b> field to query operations.
     *
     */
    public final static SalesContractItemField<String> INCOTERMS_LOCATION2 =
        new SalesContractItemField<String>("IncotermsLocation2");
    /**
     * Constraints: Not nullable, Maximum length: 4
     * <p>
     * Original property name from the Odata EDM: <b>CustomerPaymentTerms</b>
     * </p>
     *
     * @return Terms of Payment Key
     */
    @SerializedName( "CustomerPaymentTerms" )
    @JsonProperty( "CustomerPaymentTerms" )
    @Nullable
    @ODataField( odataName = "CustomerPaymentTerms" )
    private String customerPaymentTerms;
    /**
     * Use with available fluent helpers to apply the <b>CustomerPaymentTerms</b> field to query operations.
     *
     */
    public final static SalesContractItemField<String> CUSTOMER_PAYMENT_TERMS =
        new SalesContractItemField<String>("CustomerPaymentTerms");
    /**
     * Constraints: Not nullable, Maximum length: 2
     * <p>
     * Original property name from the Odata EDM: <b>SalesDocumentRjcnReason</b>
     * </p>
     *
     * @return Reason for Rejection of Sales Documents
     */
    @SerializedName( "SalesDocumentRjcnReason" )
    @JsonProperty( "SalesDocumentRjcnReason" )
    @Nullable
    @ODataField( odataName = "SalesDocumentRjcnReason" )
    private String salesDocumentRjcnReason;
    /**
     * Use with available fluent helpers to apply the <b>SalesDocumentRjcnReason</b> field to query operations.
     *
     */
    public final static SalesContractItemField<String> SALES_DOCUMENT_RJCN_REASON =
        new SalesContractItemField<String>("SalesDocumentRjcnReason");
    /**
     * Constraints: Not nullable, Maximum length: 2
     * <p>
     * Original property name from the Odata EDM: <b>ItemBillingBlockReason</b>
     * </p>
     *
     * @return Billing Block for Item
     */
    @SerializedName( "ItemBillingBlockReason" )
    @JsonProperty( "ItemBillingBlockReason" )
    @Nullable
    @ODataField( odataName = "ItemBillingBlockReason" )
    private String itemBillingBlockReason;
    /**
     * Use with available fluent helpers to apply the <b>ItemBillingBlockReason</b> field to query operations.
     *
     */
    public final static SalesContractItemField<String> ITEM_BILLING_BLOCK_REASON =
        new SalesContractItemField<String>("ItemBillingBlockReason");
    /**
     * Constraints: Not nullable, Maximum length: 24
     * <p>
     * Original property name from the Odata EDM: <b>WBSElement</b>
     * </p>
     *
     * @return Work Breakdown Structure Element (WBS Element)
     */
    @SerializedName( "WBSElement" )
    @JsonProperty( "WBSElement" )
    @Nullable
    @ODataField( odataName = "WBSElement" )
    private String wBSElement;
    /**
     * Use with available fluent helpers to apply the <b>WBSElement</b> field to query operations.
     *
     */
    public final static SalesContractItemField<String> WBS_ELEMENT = new SalesContractItemField<String>("WBSElement");
    /**
     * Constraints: Not nullable, Maximum length: 10
     * <p>
     * Original property name from the Odata EDM: <b>ProfitCenter</b>
     * </p>
     *
     * @return Profit Center
     */
    @SerializedName( "ProfitCenter" )
    @JsonProperty( "ProfitCenter" )
    @Nullable
    @ODataField( odataName = "ProfitCenter" )
    private String profitCenter;
    /**
     * Use with available fluent helpers to apply the <b>ProfitCenter</b> field to query operations.
     *
     */
    public final static SalesContractItemField<String> PROFIT_CENTER =
        new SalesContractItemField<String>("ProfitCenter");
    /**
     * Constraints: Not nullable, Maximum length: 10
     * <p>
     * Original property name from the Odata EDM: <b>ReferenceSDDocument</b>
     * </p>
     *
     * @return Document Number of Reference Document
     */
    @SerializedName( "ReferenceSDDocument" )
    @JsonProperty( "ReferenceSDDocument" )
    @Nullable
    @ODataField( odataName = "ReferenceSDDocument" )
    private String referenceSDDocument;
    /**
     * Use with available fluent helpers to apply the <b>ReferenceSDDocument</b> field to query operations.
     *
     */
    public final static SalesContractItemField<String> REFERENCE_SD_DOCUMENT =
        new SalesContractItemField<String>("ReferenceSDDocument");
    /**
     * Constraints: Not nullable, Maximum length: 6
     * <p>
     * Original property name from the Odata EDM: <b>ReferenceSDDocumentItem</b>
     * </p>
     *
     * @return Item Number of the Reference Item
     */
    @SerializedName( "ReferenceSDDocumentItem" )
    @JsonProperty( "ReferenceSDDocumentItem" )
    @Nullable
    @ODataField( odataName = "ReferenceSDDocumentItem" )
    private String referenceSDDocumentItem;
    /**
     * Use with available fluent helpers to apply the <b>ReferenceSDDocumentItem</b> field to query operations.
     *
     */
    public final static SalesContractItemField<String> REFERENCE_SD_DOCUMENT_ITEM =
        new SalesContractItemField<String>("ReferenceSDDocumentItem");
    /**
     * Constraints: Not nullable, Maximum length: 1
     * <p>
     * Original property name from the Odata EDM: <b>SDProcessStatus</b>
     * </p>
     *
     * @return Overall Processing Status (Item)
     */
    @SerializedName( "SDProcessStatus" )
    @JsonProperty( "SDProcessStatus" )
    @Nullable
    @ODataField( odataName = "SDProcessStatus" )
    private String sDProcessStatus;
    /**
     * Use with available fluent helpers to apply the <b>SDProcessStatus</b> field to query operations.
     *
     */
    public final static SalesContractItemField<String> SD_PROCESS_STATUS =
        new SalesContractItemField<String>("SDProcessStatus");
    /**
     * Constraints: Not nullable, Precision: 0
     * <p>
     * Original property name from the Odata EDM: <b>SalesContractValidityStartDate</b>
     * </p>
     *
     * @return Contract Start Date
     */
    @SerializedName( "SalesContractValidityStartDate" )
    @JsonProperty( "SalesContractValidityStartDate" )
    @Nullable
    @JsonSerialize( using = com.sap.cloud.sdk.s4hana.datamodel.odata.adapter.JacksonLocalDateTimeSerializer.class )
    @JsonDeserialize( using = com.sap.cloud.sdk.s4hana.datamodel.odata.adapter.JacksonLocalDateTimeDeserializer.class )
    @JsonAdapter( com.sap.cloud.sdk.s4hana.datamodel.odata.adapter.LocalDateTimeAdapter.class )
    @ODataField(
        odataName = "SalesContractValidityStartDate",
        converter = com.sap.cloud.sdk.s4hana.datamodel.odata.adapter.LocalDateTimeCalendarConverter.class )
    private LocalDateTime salesContractValidityStartDate;
    /**
     * Use with available fluent helpers to apply the <b>SalesContractValidityStartDate</b> field to query operations.
     *
     */
    public final static SalesContractItemField<LocalDateTime> SALES_CONTRACT_VALIDITY_START_DATE =
        new SalesContractItemField<LocalDateTime>("SalesContractValidityStartDate");
    /**
     * Constraints: Not nullable, Precision: 0
     * <p>
     * Original property name from the Odata EDM: <b>SalesContractValidityEndDate</b>
     * </p>
     *
     * @return Contract End Date
     */
    @SerializedName( "SalesContractValidityEndDate" )
    @JsonProperty( "SalesContractValidityEndDate" )
    @Nullable
    @JsonSerialize( using = com.sap.cloud.sdk.s4hana.datamodel.odata.adapter.JacksonLocalDateTimeSerializer.class )
    @JsonDeserialize( using = com.sap.cloud.sdk.s4hana.datamodel.odata.adapter.JacksonLocalDateTimeDeserializer.class )
    @JsonAdapter( com.sap.cloud.sdk.s4hana.datamodel.odata.adapter.LocalDateTimeAdapter.class )
    @ODataField(
        odataName = "SalesContractValidityEndDate",
        converter = com.sap.cloud.sdk.s4hana.datamodel.odata.adapter.LocalDateTimeCalendarConverter.class )
    private LocalDateTime salesContractValidityEndDate;
    /**
     * Use with available fluent helpers to apply the <b>SalesContractValidityEndDate</b> field to query operations.
     *
     */
    public final static SalesContractItemField<LocalDateTime> SALES_CONTRACT_VALIDITY_END_DATE =
        new SalesContractItemField<LocalDateTime>("SalesContractValidityEndDate");
    /**
     * Constraints: Not nullable, Maximum length: 3
     * <p>
     * Original property name from the Odata EDM: <b>NmbrOfSalesContractValdtyPerd</b>
     * </p>
     *
     * @return Validity Period of Contract
     */
    @SerializedName( "NmbrOfSalesContractValdtyPerd" )
    @JsonProperty( "NmbrOfSalesContractValdtyPerd" )
    @Nullable
    @ODataField( odataName = "NmbrOfSalesContractValdtyPerd" )
    private String nmbrOfSalesContractValdtyPerd;
    /**
     * Use with available fluent helpers to apply the <b>NmbrOfSalesContractValdtyPerd</b> field to query operations.
     *
     */
    public final static SalesContractItemField<String> NMBR_OF_SALES_CONTRACT_VALDTY_PERD =
        new SalesContractItemField<String>("NmbrOfSalesContractValdtyPerd");
    /**
     * Constraints: Not nullable, Maximum length: 1
     * <p>
     * Original property name from the Odata EDM: <b>SalesContractValidityPerdUnit</b>
     * </p>
     *
     * @return Unit of Validity Period of Contract
     */
    @SerializedName( "SalesContractValidityPerdUnit" )
    @JsonProperty( "SalesContractValidityPerdUnit" )
    @Nullable
    @ODataField( odataName = "SalesContractValidityPerdUnit" )
    private String salesContractValidityPerdUnit;
    /**
     * Use with available fluent helpers to apply the <b>SalesContractValidityPerdUnit</b> field to query operations.
     *
     */
    public final static SalesContractItemField<String> SALES_CONTRACT_VALIDITY_PERD_UNIT =
        new SalesContractItemField<String>("SalesContractValidityPerdUnit");
    /**
     * Constraints: Not nullable, Maximum length: 2
     * <p>
     * Original property name from the Odata EDM: <b>SalesContractValidityPerdCat</b>
     * </p>
     *
     * @return Validity Period Category of Contract
     */
    @SerializedName( "SalesContractValidityPerdCat" )
    @JsonProperty( "SalesContractValidityPerdCat" )
    @Nullable
    @ODataField( odataName = "SalesContractValidityPerdCat" )
    private String salesContractValidityPerdCat;
    /**
     * Use with available fluent helpers to apply the <b>SalesContractValidityPerdCat</b> field to query operations.
     *
     */
    public final static SalesContractItemField<String> SALES_CONTRACT_VALIDITY_PERD_CAT =
        new SalesContractItemField<String>("SalesContractValidityPerdCat");
    /**
     * Constraints: Not nullable, Precision: 0
     * <p>
     * Original property name from the Odata EDM: <b>SalesContractSignedDate</b>
     * </p>
     *
     * @return Date on Which Contract Is Signed
     */
    @SerializedName( "SalesContractSignedDate" )
    @JsonProperty( "SalesContractSignedDate" )
    @Nullable
    @JsonSerialize( using = com.sap.cloud.sdk.s4hana.datamodel.odata.adapter.JacksonLocalDateTimeSerializer.class )
    @JsonDeserialize( using = com.sap.cloud.sdk.s4hana.datamodel.odata.adapter.JacksonLocalDateTimeDeserializer.class )
    @JsonAdapter( com.sap.cloud.sdk.s4hana.datamodel.odata.adapter.LocalDateTimeAdapter.class )
    @ODataField(
        odataName = "SalesContractSignedDate",
        converter = com.sap.cloud.sdk.s4hana.datamodel.odata.adapter.LocalDateTimeCalendarConverter.class )
    private LocalDateTime salesContractSignedDate;
    /**
     * Use with available fluent helpers to apply the <b>SalesContractSignedDate</b> field to query operations.
     *
     */
    public final static SalesContractItemField<LocalDateTime> SALES_CONTRACT_SIGNED_DATE =
        new SalesContractItemField<LocalDateTime>("SalesContractSignedDate");
    /**
     * Constraints: Not nullable, Maximum length: 4
     * <p>
     * Original property name from the Odata EDM: <b>SalesContractFollowUpAction</b>
     * </p>
     *
     * @return Action at end of contract
     */
    @SerializedName( "SalesContractFollowUpAction" )
    @JsonProperty( "SalesContractFollowUpAction" )
    @Nullable
    @ODataField( odataName = "SalesContractFollowUpAction" )
    private String salesContractFollowUpAction;
    /**
     * Use with available fluent helpers to apply the <b>SalesContractFollowUpAction</b> field to query operations.
     *
     */
    public final static SalesContractItemField<String> SALES_CONTRACT_FOLLOW_UP_ACTION =
        new SalesContractItemField<String>("SalesContractFollowUpAction");
    /**
     * Constraints: Not nullable, Precision: 0
     * <p>
     * Original property name from the Odata EDM: <b>SlsContractFollowUpActionDate</b>
     * </p>
     *
     * @return Date for Action
     */
    @SerializedName( "SlsContractFollowUpActionDate" )
    @JsonProperty( "SlsContractFollowUpActionDate" )
    @Nullable
    @JsonSerialize( using = com.sap.cloud.sdk.s4hana.datamodel.odata.adapter.JacksonLocalDateTimeSerializer.class )
    @JsonDeserialize( using = com.sap.cloud.sdk.s4hana.datamodel.odata.adapter.JacksonLocalDateTimeDeserializer.class )
    @JsonAdapter( com.sap.cloud.sdk.s4hana.datamodel.odata.adapter.LocalDateTimeAdapter.class )
    @ODataField(
        odataName = "SlsContractFollowUpActionDate",
        converter = com.sap.cloud.sdk.s4hana.datamodel.odata.adapter.LocalDateTimeCalendarConverter.class )
    private LocalDateTime slsContractFollowUpActionDate;
    /**
     * Use with available fluent helpers to apply the <b>SlsContractFollowUpActionDate</b> field to query operations.
     *
     */
    public final static SalesContractItemField<LocalDateTime> SLS_CONTRACT_FOLLOW_UP_ACTION_DATE =
        new SalesContractItemField<LocalDateTime>("SlsContractFollowUpActionDate");
    /**
     * Constraints: Not nullable, Precision: 15, Scale: 3
     * <p>
     * Original property name from the Odata EDM: <b>SlsContrItemReleasedQuantity</b>
     * </p>
     *
     * @return Released Quantity on Sales Contract Item
     */
    @SerializedName( "SlsContrItemReleasedQuantity" )
    @JsonProperty( "SlsContrItemReleasedQuantity" )
    @Nullable
    @ODataField( odataName = "SlsContrItemReleasedQuantity" )
    private BigDecimal slsContrItemReleasedQuantity;
    /**
     * Use with available fluent helpers to apply the <b>SlsContrItemReleasedQuantity</b> field to query operations.
     *
     */
    public final static SalesContractItemField<BigDecimal> SLS_CONTR_ITEM_RELEASED_QUANTITY =
        new SalesContractItemField<BigDecimal>("SlsContrItemReleasedQuantity");
    /**
     * Constraints: Not nullable, Maximum length: 3
     * <p>
     * Original property name from the Odata EDM: <b>SlsContrItmReldQuantityUnit</b>
     * </p>
     *
     * @return Released Quantity Unit on Sales Contract Item
     */
    @SerializedName( "SlsContrItmReldQuantityUnit" )
    @JsonProperty( "SlsContrItmReldQuantityUnit" )
    @Nullable
    @ODataField( odataName = "SlsContrItmReldQuantityUnit" )
    private String slsContrItmReldQuantityUnit;
    /**
     * Use with available fluent helpers to apply the <b>SlsContrItmReldQuantityUnit</b> field to query operations.
     *
     */
    public final static SalesContractItemField<String> SLS_CONTR_ITM_RELD_QUANTITY_UNIT =
        new SalesContractItemField<String>("SlsContrItmReldQuantityUnit");
    /**
     * Navigation property <b>to_Partner</b> for <b>SalesContractItem</b> to multiple <b>SalesContractItemPartner</b>.
     *
     */
    @SerializedName( "to_Partner" )
    @JsonProperty( "to_Partner" )
    @ODataField( odataName = "to_Partner" )
    @Getter( AccessLevel.NONE )
    @Setter( AccessLevel.NONE )
    private List<SalesContractItemPartner> toPartner;
    /**
     * Navigation property <b>to_PricingElement</b> for <b>SalesContractItem</b> to multiple
     * <b>SalesContractItemPrcgElmnt</b>.
     *
     */
    @SerializedName( "to_PricingElement" )
    @JsonProperty( "to_PricingElement" )
    @ODataField( odataName = "to_PricingElement" )
    @Getter( AccessLevel.NONE )
    @Setter( AccessLevel.NONE )
    private List<SalesContractItemPrcgElmnt> toPricingElement;
    /**
     * Navigation property <b>to_SalesContract</b> for <b>SalesContractItem</b> to single <b>SalesContract</b>.
     *
     */
    @SerializedName( "to_SalesContract" )
    @JsonProperty( "to_SalesContract" )
    @ODataField( odataName = "to_SalesContract" )
    @Nullable
    @Getter( AccessLevel.NONE )
    @Setter( AccessLevel.NONE )
    private SalesContract toSalesContract;
    /**
     * Navigation property <b>to_Text</b> for <b>SalesContractItem</b> to multiple <b>SalesContractItemText</b>.
     *
     */
    @SerializedName( "to_Text" )
    @JsonProperty( "to_Text" )
    @ODataField( odataName = "to_Text" )
    @Getter( AccessLevel.NONE )
    @Setter( AccessLevel.NONE )
    private List<SalesContractItemText> toText;
    /**
     * Use with available fluent helpers to apply the <b>to_Partner</b> navigation property to query operations.
     *
     */
    public final static SalesContractItemLink<SalesContractItemPartner> TO_PARTNER =
        new SalesContractItemLink<SalesContractItemPartner>("to_Partner");
    /**
     * Use with available fluent helpers to apply the <b>to_PricingElement</b> navigation property to query operations.
     *
     */
    public final static SalesContractItemLink<SalesContractItemPrcgElmnt> TO_PRICING_ELEMENT =
        new SalesContractItemLink<SalesContractItemPrcgElmnt>("to_PricingElement");
    /**
     * Use with available fluent helpers to apply the <b>to_SalesContract</b> navigation property to query operations.
     *
     */
    public final static SalesContractItemOneToOneLink<SalesContract> TO_SALES_CONTRACT =
        new SalesContractItemOneToOneLink<SalesContract>("to_SalesContract");
    /**
     * Use with available fluent helpers to apply the <b>to_Text</b> navigation property to query operations.
     *
     */
    public final static SalesContractItemLink<SalesContractItemText> TO_TEXT =
        new SalesContractItemLink<SalesContractItemText>("to_Text");

    @Nonnull
    @Override
    public Class<SalesContractItem> getType()
    {
        return SalesContractItem.class;
    }

    /**
     * (Key Field) Constraints: Not nullable, Maximum length: 10
     * <p>
     * Original property name from the Odata EDM: <b>SalesContract</b>
     * </p>
     *
     * @param salesContract
     *            Sales Contract
     */
    public void setSalesContract( @Nullable final String salesContract )
    {
        rememberChangedField("SalesContract", this.salesContract);
        this.salesContract = salesContract;
    }

    /**
     * (Key Field) Constraints: Not nullable, Maximum length: 6
     * <p>
     * Original property name from the Odata EDM: <b>SalesContractItem</b>
     * </p>
     *
     * @param salesContractItem
     *            Sales Contract Item
     */
    public void setSalesContractItem( @Nullable final String salesContractItem )
    {
        rememberChangedField("SalesContractItem", this.salesContractItem);
        this.salesContractItem = salesContractItem;
    }

    /**
     * Constraints: Not nullable, Maximum length: 6
     * <p>
     * Original property name from the Odata EDM: <b>HigherLevelItem</b>
     * </p>
     *
     * @param higherLevelItem
     *            Higher-Level Item in Bill of Material Structures
     */
    public void setHigherLevelItem( @Nullable final String higherLevelItem )
    {
        rememberChangedField("HigherLevelItem", this.higherLevelItem);
        this.higherLevelItem = higherLevelItem;
    }

    /**
     * Constraints: Not nullable, Maximum length: 4
     * <p>
     * Original property name from the Odata EDM: <b>SalesContractItemCategory</b>
     * </p>
     *
     * @param salesContractItemCategory
     *            Sales Document Item Category
     */
    public void setSalesContractItemCategory( @Nullable final String salesContractItemCategory )
    {
        rememberChangedField("SalesContractItemCategory", this.salesContractItemCategory);
        this.salesContractItemCategory = salesContractItemCategory;
    }

    /**
     * Constraints: Not nullable, Maximum length: 40
     * <p>
     * Original property name from the Odata EDM: <b>SalesContractItemText</b>
     * </p>
     *
     * @param salesContractItemText
     *            Short Text for Sales Order Item
     */
    public void setSalesContractItemText( @Nullable final String salesContractItemText )
    {
        rememberChangedField("SalesContractItemText", this.salesContractItemText);
        this.salesContractItemText = salesContractItemText;
    }

    /**
     * Constraints: Not nullable, Maximum length: 35
     * <p>
     * Original property name from the Odata EDM: <b>PurchaseOrderByCustomer</b>
     * </p>
     *
     * @param purchaseOrderByCustomer
     *            Customer Reference
     */
    public void setPurchaseOrderByCustomer( @Nullable final String purchaseOrderByCustomer )
    {
        rememberChangedField("PurchaseOrderByCustomer", this.purchaseOrderByCustomer);
        this.purchaseOrderByCustomer = purchaseOrderByCustomer;
    }

    /**
     * Constraints: Not nullable, Maximum length: 40
     * <p>
     * Original property name from the Odata EDM: <b>Material</b>
     * </p>
     *
     * @param material
     *            Material Number
     */
    public void setMaterial( @Nullable final String material )
    {
        rememberChangedField("Material", this.material);
        this.material = material;
    }

    /**
     * Constraints: Not nullable, Maximum length: 35
     * <p>
     * Original property name from the Odata EDM: <b>MaterialByCustomer</b>
     * </p>
     *
     * @param materialByCustomer
     *            Material Number Used by Customer
     */
    public void setMaterialByCustomer( @Nullable final String materialByCustomer )
    {
        rememberChangedField("MaterialByCustomer", this.materialByCustomer);
        this.materialByCustomer = materialByCustomer;
    }

    /**
     * Constraints: Not nullable, Precision: 0
     * <p>
     * Original property name from the Odata EDM: <b>PricingDate</b>
     * </p>
     *
     * @param pricingDate
     *            Date for Pricing and Exchange Rate
     */
    public void setPricingDate( @Nullable final LocalDateTime pricingDate )
    {
        rememberChangedField("PricingDate", this.pricingDate);
        this.pricingDate = pricingDate;
    }

    /**
     * Constraints: Not nullable, Precision: 15, Scale: 3
     * <p>
     * Original property name from the Odata EDM: <b>RequestedQuantity</b>
     * </p>
     *
     * @param requestedQuantity
     *            Requested Quantity
     */
    public void setRequestedQuantity( @Nullable final BigDecimal requestedQuantity )
    {
        rememberChangedField("RequestedQuantity", this.requestedQuantity);
        this.requestedQuantity = requestedQuantity;
    }

    /**
     * Constraints: Not nullable, Maximum length: 3
     * <p>
     * Original property name from the Odata EDM: <b>RequestedQuantityUnit</b>
     * </p>
     *
     * @param requestedQuantityUnit
     *            Unit of the Requested Quantity
     */
    public void setRequestedQuantityUnit( @Nullable final String requestedQuantityUnit )
    {
        rememberChangedField("RequestedQuantityUnit", this.requestedQuantityUnit);
        this.requestedQuantityUnit = requestedQuantityUnit;
    }

    /**
     * Constraints: Not nullable, Maximum length: 3
     * <p>
     * Original property name from the Odata EDM: <b>RequestedQuantitySAPUnit</b>
     * </p>
     *
     * @param requestedQuantitySAPUnit
     *            SAP Unit Code for Requested Quantity
     */
    public void setRequestedQuantitySAPUnit( @Nullable final String requestedQuantitySAPUnit )
    {
        rememberChangedField("RequestedQuantitySAPUnit", this.requestedQuantitySAPUnit);
        this.requestedQuantitySAPUnit = requestedQuantitySAPUnit;
    }

    /**
     * Constraints: Not nullable, Maximum length: 3
     * <p>
     * Original property name from the Odata EDM: <b>RequestedQuantityISOUnit</b>
     * </p>
     *
     * @param requestedQuantityISOUnit
     *            ISO Unit Code for Requested Quantity
     */
    public void setRequestedQuantityISOUnit( @Nullable final String requestedQuantityISOUnit )
    {
        rememberChangedField("RequestedQuantityISOUnit", this.requestedQuantityISOUnit);
        this.requestedQuantityISOUnit = requestedQuantityISOUnit;
    }

    /**
     * Constraints: Not nullable, Precision: 15, Scale: 3
     * <p>
     * Original property name from the Odata EDM: <b>ItemGrossWeight</b>
     * </p>
     *
     * @param itemGrossWeight
     *            Gross Weight of the Item
     */
    public void setItemGrossWeight( @Nullable final BigDecimal itemGrossWeight )
    {
        rememberChangedField("ItemGrossWeight", this.itemGrossWeight);
        this.itemGrossWeight = itemGrossWeight;
    }

    /**
     * Constraints: Not nullable, Precision: 15, Scale: 3
     * <p>
     * Original property name from the Odata EDM: <b>ItemNetWeight</b>
     * </p>
     *
     * @param itemNetWeight
     *            Net Weight of the Item
     */
    public void setItemNetWeight( @Nullable final BigDecimal itemNetWeight )
    {
        rememberChangedField("ItemNetWeight", this.itemNetWeight);
        this.itemNetWeight = itemNetWeight;
    }

    /**
     * Constraints: Not nullable, Maximum length: 3
     * <p>
     * Original property name from the Odata EDM: <b>ItemWeightUnit</b>
     * </p>
     *
     * @param itemWeightUnit
     *            Unit of Weight
     */
    public void setItemWeightUnit( @Nullable final String itemWeightUnit )
    {
        rememberChangedField("ItemWeightUnit", this.itemWeightUnit);
        this.itemWeightUnit = itemWeightUnit;
    }

    /**
     * Constraints: Not nullable, Maximum length: 3
     * <p>
     * Original property name from the Odata EDM: <b>ItemWeightSAPUnit</b>
     * </p>
     *
     * @param itemWeightSAPUnit
     *            SAP Unit Code for Item Weight
     */
    public void setItemWeightSAPUnit( @Nullable final String itemWeightSAPUnit )
    {
        rememberChangedField("ItemWeightSAPUnit", this.itemWeightSAPUnit);
        this.itemWeightSAPUnit = itemWeightSAPUnit;
    }

    /**
     * Constraints: Not nullable, Maximum length: 3
     * <p>
     * Original property name from the Odata EDM: <b>ItemWeightISOUnit</b>
     * </p>
     *
     * @param itemWeightISOUnit
     *            ISO Unit Code for Item Weight
     */
    public void setItemWeightISOUnit( @Nullable final String itemWeightISOUnit )
    {
        rememberChangedField("ItemWeightISOUnit", this.itemWeightISOUnit);
        this.itemWeightISOUnit = itemWeightISOUnit;
    }

    /**
     * Constraints: Not nullable, Precision: 15, Scale: 3
     * <p>
     * Original property name from the Odata EDM: <b>ItemVolume</b>
     * </p>
     *
     * @param itemVolume
     *            Volume of the item
     */
    public void setItemVolume( @Nullable final BigDecimal itemVolume )
    {
        rememberChangedField("ItemVolume", this.itemVolume);
        this.itemVolume = itemVolume;
    }

    /**
     * Constraints: Not nullable, Maximum length: 3
     * <p>
     * Original property name from the Odata EDM: <b>ItemVolumeUnit</b>
     * </p>
     *
     * @param itemVolumeUnit
     *            Volume Unit
     */
    public void setItemVolumeUnit( @Nullable final String itemVolumeUnit )
    {
        rememberChangedField("ItemVolumeUnit", this.itemVolumeUnit);
        this.itemVolumeUnit = itemVolumeUnit;
    }

    /**
     * Constraints: Not nullable, Maximum length: 3
     * <p>
     * Original property name from the Odata EDM: <b>ItemVolumeSAPUnit</b>
     * </p>
     *
     * @param itemVolumeSAPUnit
     *            SAP Unit Code for Item Volume
     */
    public void setItemVolumeSAPUnit( @Nullable final String itemVolumeSAPUnit )
    {
        rememberChangedField("ItemVolumeSAPUnit", this.itemVolumeSAPUnit);
        this.itemVolumeSAPUnit = itemVolumeSAPUnit;
    }

    /**
     * Constraints: Not nullable, Maximum length: 3
     * <p>
     * Original property name from the Odata EDM: <b>ItemVolumeISOUnit</b>
     * </p>
     *
     * @param itemVolumeISOUnit
     *            ISO Unit Code for Item Volume
     */
    public void setItemVolumeISOUnit( @Nullable final String itemVolumeISOUnit )
    {
        rememberChangedField("ItemVolumeISOUnit", this.itemVolumeISOUnit);
        this.itemVolumeISOUnit = itemVolumeISOUnit;
    }

    /**
     * Constraints: Not nullable, Precision: 14, Scale: 3
     * <p>
     * Original property name from the Odata EDM: <b>OutlineAgreementTargetAmount</b>
     * </p>
     *
     * @param outlineAgreementTargetAmount
     *            Target Value for Outline Agreement in Document Currency
     */
    public void setOutlineAgreementTargetAmount( @Nullable final BigDecimal outlineAgreementTargetAmount )
    {
        rememberChangedField("OutlineAgreementTargetAmount", this.outlineAgreementTargetAmount);
        this.outlineAgreementTargetAmount = outlineAgreementTargetAmount;
    }

    /**
     * Constraints: Not nullable, Maximum length: 5
     * <p>
     * Original property name from the Odata EDM: <b>TransactionCurrency</b>
     * </p>
     *
     * @param transactionCurrency
     *            SD Document Currency
     */
    public void setTransactionCurrency( @Nullable final String transactionCurrency )
    {
        rememberChangedField("TransactionCurrency", this.transactionCurrency);
        this.transactionCurrency = transactionCurrency;
    }

    /**
     * Constraints: Not nullable, Precision: 16, Scale: 3
     * <p>
     * Original property name from the Odata EDM: <b>NetAmount</b>
     * </p>
     *
     * @param netAmount
     *            Net Value of the Order Item in Document Currency
     */
    public void setNetAmount( @Nullable final BigDecimal netAmount )
    {
        rememberChangedField("NetAmount", this.netAmount);
        this.netAmount = netAmount;
    }

    /**
     * Constraints: Not nullable, Maximum length: 9
     * <p>
     * Original property name from the Odata EDM: <b>MaterialGroup</b>
     * </p>
     *
     * @param materialGroup
     *            Material Group
     */
    public void setMaterialGroup( @Nullable final String materialGroup )
    {
        rememberChangedField("MaterialGroup", this.materialGroup);
        this.materialGroup = materialGroup;
    }

    /**
     * Constraints: Not nullable, Maximum length: 2
     * <p>
     * Original property name from the Odata EDM: <b>MaterialPricingGroup</b>
     * </p>
     *
     * @param materialPricingGroup
     *            Material Price Group
     */
    public void setMaterialPricingGroup( @Nullable final String materialPricingGroup )
    {
        rememberChangedField("MaterialPricingGroup", this.materialPricingGroup);
        this.materialPricingGroup = materialPricingGroup;
    }

    /**
     * Constraints: Not nullable, Maximum length: 10
     * <p>
     * Original property name from the Odata EDM: <b>Batch</b>
     * </p>
     *
     * @param batch
     *            Batch Number
     */
    public void setBatch( @Nullable final String batch )
    {
        rememberChangedField("Batch", this.batch);
        this.batch = batch;
    }

    /**
     * Constraints: Not nullable, Maximum length: 4
     * <p>
     * Original property name from the Odata EDM: <b>ProductionPlant</b>
     * </p>
     *
     * @param productionPlant
     *            Plant (Own or External)
     */
    public void setProductionPlant( @Nullable final String productionPlant )
    {
        rememberChangedField("ProductionPlant", this.productionPlant);
        this.productionPlant = productionPlant;
    }

    /**
     * Constraints: Not nullable, Maximum length: 4
     * <p>
     * Original property name from the Odata EDM: <b>StorageLocation</b>
     * </p>
     *
     * @param storageLocation
     *            Storage Location
     */
    public void setStorageLocation( @Nullable final String storageLocation )
    {
        rememberChangedField("StorageLocation", this.storageLocation);
        this.storageLocation = storageLocation;
    }

    /**
     * Constraints: Not nullable, Maximum length: 4
     * <p>
     * Original property name from the Odata EDM: <b>ShippingPoint</b>
     * </p>
     *
     * @param shippingPoint
     *            Shipping Point / Receiving Point
     */
    public void setShippingPoint( @Nullable final String shippingPoint )
    {
        rememberChangedField("ShippingPoint", this.shippingPoint);
        this.shippingPoint = shippingPoint;
    }

    /**
     * Constraints: Not nullable, Maximum length: 3
     * <p>
     * Original property name from the Odata EDM: <b>IncotermsClassification</b>
     * </p>
     *
     * @param incotermsClassification
     *            Incoterms (Part 1)
     */
    public void setIncotermsClassification( @Nullable final String incotermsClassification )
    {
        rememberChangedField("IncotermsClassification", this.incotermsClassification);
        this.incotermsClassification = incotermsClassification;
    }

    /**
     * Constraints: Not nullable, Maximum length: 28
     * <p>
     * Original property name from the Odata EDM: <b>IncotermsTransferLocation</b>
     * </p>
     *
     * @param incotermsTransferLocation
     *            Incoterms (Part 2)
     */
    public void setIncotermsTransferLocation( @Nullable final String incotermsTransferLocation )
    {
        rememberChangedField("IncotermsTransferLocation", this.incotermsTransferLocation);
        this.incotermsTransferLocation = incotermsTransferLocation;
    }

    /**
     * Constraints: Not nullable, Maximum length: 70
     * <p>
     * Original property name from the Odata EDM: <b>IncotermsLocation1</b>
     * </p>
     *
     * @param incotermsLocation1
     *            Incoterms Location 1
     */
    public void setIncotermsLocation1( @Nullable final String incotermsLocation1 )
    {
        rememberChangedField("IncotermsLocation1", this.incotermsLocation1);
        this.incotermsLocation1 = incotermsLocation1;
    }

    /**
     * Constraints: Not nullable, Maximum length: 70
     * <p>
     * Original property name from the Odata EDM: <b>IncotermsLocation2</b>
     * </p>
     *
     * @param incotermsLocation2
     *            Incoterms Location 2
     */
    public void setIncotermsLocation2( @Nullable final String incotermsLocation2 )
    {
        rememberChangedField("IncotermsLocation2", this.incotermsLocation2);
        this.incotermsLocation2 = incotermsLocation2;
    }

    /**
     * Constraints: Not nullable, Maximum length: 4
     * <p>
     * Original property name from the Odata EDM: <b>CustomerPaymentTerms</b>
     * </p>
     *
     * @param customerPaymentTerms
     *            Terms of Payment Key
     */
    public void setCustomerPaymentTerms( @Nullable final String customerPaymentTerms )
    {
        rememberChangedField("CustomerPaymentTerms", this.customerPaymentTerms);
        this.customerPaymentTerms = customerPaymentTerms;
    }

    /**
     * Constraints: Not nullable, Maximum length: 2
     * <p>
     * Original property name from the Odata EDM: <b>SalesDocumentRjcnReason</b>
     * </p>
     *
     * @param salesDocumentRjcnReason
     *            Reason for Rejection of Sales Documents
     */
    public void setSalesDocumentRjcnReason( @Nullable final String salesDocumentRjcnReason )
    {
        rememberChangedField("SalesDocumentRjcnReason", this.salesDocumentRjcnReason);
        this.salesDocumentRjcnReason = salesDocumentRjcnReason;
    }

    /**
     * Constraints: Not nullable, Maximum length: 2
     * <p>
     * Original property name from the Odata EDM: <b>ItemBillingBlockReason</b>
     * </p>
     *
     * @param itemBillingBlockReason
     *            Billing Block for Item
     */
    public void setItemBillingBlockReason( @Nullable final String itemBillingBlockReason )
    {
        rememberChangedField("ItemBillingBlockReason", this.itemBillingBlockReason);
        this.itemBillingBlockReason = itemBillingBlockReason;
    }

    /**
     * Constraints: Not nullable, Maximum length: 24
     * <p>
     * Original property name from the Odata EDM: <b>WBSElement</b>
     * </p>
     *
     * @param wBSElement
     *            Work Breakdown Structure Element (WBS Element)
     */
    public void setWBSElement( @Nullable final String wBSElement )
    {
        rememberChangedField("WBSElement", this.wBSElement);
        this.wBSElement = wBSElement;
    }

    /**
     * Constraints: Not nullable, Maximum length: 10
     * <p>
     * Original property name from the Odata EDM: <b>ProfitCenter</b>
     * </p>
     *
     * @param profitCenter
     *            Profit Center
     */
    public void setProfitCenter( @Nullable final String profitCenter )
    {
        rememberChangedField("ProfitCenter", this.profitCenter);
        this.profitCenter = profitCenter;
    }

    /**
     * Constraints: Not nullable, Maximum length: 10
     * <p>
     * Original property name from the Odata EDM: <b>ReferenceSDDocument</b>
     * </p>
     *
     * @param referenceSDDocument
     *            Document Number of Reference Document
     */
    public void setReferenceSDDocument( @Nullable final String referenceSDDocument )
    {
        rememberChangedField("ReferenceSDDocument", this.referenceSDDocument);
        this.referenceSDDocument = referenceSDDocument;
    }

    /**
     * Constraints: Not nullable, Maximum length: 6
     * <p>
     * Original property name from the Odata EDM: <b>ReferenceSDDocumentItem</b>
     * </p>
     *
     * @param referenceSDDocumentItem
     *            Item Number of the Reference Item
     */
    public void setReferenceSDDocumentItem( @Nullable final String referenceSDDocumentItem )
    {
        rememberChangedField("ReferenceSDDocumentItem", this.referenceSDDocumentItem);
        this.referenceSDDocumentItem = referenceSDDocumentItem;
    }

    /**
     * Constraints: Not nullable, Maximum length: 1
     * <p>
     * Original property name from the Odata EDM: <b>SDProcessStatus</b>
     * </p>
     *
     * @param sDProcessStatus
     *            Overall Processing Status (Item)
     */
    public void setSDProcessStatus( @Nullable final String sDProcessStatus )
    {
        rememberChangedField("SDProcessStatus", this.sDProcessStatus);
        this.sDProcessStatus = sDProcessStatus;
    }

    /**
     * Constraints: Not nullable, Precision: 0
     * <p>
     * Original property name from the Odata EDM: <b>SalesContractValidityStartDate</b>
     * </p>
     *
     * @param salesContractValidityStartDate
     *            Contract Start Date
     */
    public void setSalesContractValidityStartDate( @Nullable final LocalDateTime salesContractValidityStartDate )
    {
        rememberChangedField("SalesContractValidityStartDate", this.salesContractValidityStartDate);
        this.salesContractValidityStartDate = salesContractValidityStartDate;
    }

    /**
     * Constraints: Not nullable, Precision: 0
     * <p>
     * Original property name from the Odata EDM: <b>SalesContractValidityEndDate</b>
     * </p>
     *
     * @param salesContractValidityEndDate
     *            Contract End Date
     */
    public void setSalesContractValidityEndDate( @Nullable final LocalDateTime salesContractValidityEndDate )
    {
        rememberChangedField("SalesContractValidityEndDate", this.salesContractValidityEndDate);
        this.salesContractValidityEndDate = salesContractValidityEndDate;
    }

    /**
     * Constraints: Not nullable, Maximum length: 3
     * <p>
     * Original property name from the Odata EDM: <b>NmbrOfSalesContractValdtyPerd</b>
     * </p>
     *
     * @param nmbrOfSalesContractValdtyPerd
     *            Validity Period of Contract
     */
    public void setNmbrOfSalesContractValdtyPerd( @Nullable final String nmbrOfSalesContractValdtyPerd )
    {
        rememberChangedField("NmbrOfSalesContractValdtyPerd", this.nmbrOfSalesContractValdtyPerd);
        this.nmbrOfSalesContractValdtyPerd = nmbrOfSalesContractValdtyPerd;
    }

    /**
     * Constraints: Not nullable, Maximum length: 1
     * <p>
     * Original property name from the Odata EDM: <b>SalesContractValidityPerdUnit</b>
     * </p>
     *
     * @param salesContractValidityPerdUnit
     *            Unit of Validity Period of Contract
     */
    public void setSalesContractValidityPerdUnit( @Nullable final String salesContractValidityPerdUnit )
    {
        rememberChangedField("SalesContractValidityPerdUnit", this.salesContractValidityPerdUnit);
        this.salesContractValidityPerdUnit = salesContractValidityPerdUnit;
    }

    /**
     * Constraints: Not nullable, Maximum length: 2
     * <p>
     * Original property name from the Odata EDM: <b>SalesContractValidityPerdCat</b>
     * </p>
     *
     * @param salesContractValidityPerdCat
     *            Validity Period Category of Contract
     */
    public void setSalesContractValidityPerdCat( @Nullable final String salesContractValidityPerdCat )
    {
        rememberChangedField("SalesContractValidityPerdCat", this.salesContractValidityPerdCat);
        this.salesContractValidityPerdCat = salesContractValidityPerdCat;
    }

    /**
     * Constraints: Not nullable, Precision: 0
     * <p>
     * Original property name from the Odata EDM: <b>SalesContractSignedDate</b>
     * </p>
     *
     * @param salesContractSignedDate
     *            Date on Which Contract Is Signed
     */
    public void setSalesContractSignedDate( @Nullable final LocalDateTime salesContractSignedDate )
    {
        rememberChangedField("SalesContractSignedDate", this.salesContractSignedDate);
        this.salesContractSignedDate = salesContractSignedDate;
    }

    /**
     * Constraints: Not nullable, Maximum length: 4
     * <p>
     * Original property name from the Odata EDM: <b>SalesContractFollowUpAction</b>
     * </p>
     *
     * @param salesContractFollowUpAction
     *            Action at end of contract
     */
    public void setSalesContractFollowUpAction( @Nullable final String salesContractFollowUpAction )
    {
        rememberChangedField("SalesContractFollowUpAction", this.salesContractFollowUpAction);
        this.salesContractFollowUpAction = salesContractFollowUpAction;
    }

    /**
     * Constraints: Not nullable, Precision: 0
     * <p>
     * Original property name from the Odata EDM: <b>SlsContractFollowUpActionDate</b>
     * </p>
     *
     * @param slsContractFollowUpActionDate
     *            Date for Action
     */
    public void setSlsContractFollowUpActionDate( @Nullable final LocalDateTime slsContractFollowUpActionDate )
    {
        rememberChangedField("SlsContractFollowUpActionDate", this.slsContractFollowUpActionDate);
        this.slsContractFollowUpActionDate = slsContractFollowUpActionDate;
    }

    /**
     * Constraints: Not nullable, Precision: 15, Scale: 3
     * <p>
     * Original property name from the Odata EDM: <b>SlsContrItemReleasedQuantity</b>
     * </p>
     *
     * @param slsContrItemReleasedQuantity
     *            Released Quantity on Sales Contract Item
     */
    public void setSlsContrItemReleasedQuantity( @Nullable final BigDecimal slsContrItemReleasedQuantity )
    {
        rememberChangedField("SlsContrItemReleasedQuantity", this.slsContrItemReleasedQuantity);
        this.slsContrItemReleasedQuantity = slsContrItemReleasedQuantity;
    }

    /**
     * Constraints: Not nullable, Maximum length: 3
     * <p>
     * Original property name from the Odata EDM: <b>SlsContrItmReldQuantityUnit</b>
     * </p>
     *
     * @param slsContrItmReldQuantityUnit
     *            Released Quantity Unit on Sales Contract Item
     */
    public void setSlsContrItmReldQuantityUnit( @Nullable final String slsContrItmReldQuantityUnit )
    {
        rememberChangedField("SlsContrItmReldQuantityUnit", this.slsContrItmReldQuantityUnit);
        this.slsContrItmReldQuantityUnit = slsContrItmReldQuantityUnit;
    }

    @Override
    protected String getEntityCollection()
    {
        return "A_SalesContractItem";
    }

    @Nonnull
    @Override
    protected Map<String, Object> getKey()
    {
        final Map<String, Object> result = Maps.newHashMap();
        result.put("SalesContract", getSalesContract());
        result.put("SalesContractItem", getSalesContractItem());
        return result;
    }

    @Nonnull
    @Override
    protected Map<String, Object> toMapOfFields()
    {
        final Map<String, Object> values = super.toMapOfFields();
        values.put("SalesContract", getSalesContract());
        values.put("SalesContractItem", getSalesContractItem());
        values.put("HigherLevelItem", getHigherLevelItem());
        values.put("SalesContractItemCategory", getSalesContractItemCategory());
        values.put("SalesContractItemText", getSalesContractItemText());
        values.put("PurchaseOrderByCustomer", getPurchaseOrderByCustomer());
        values.put("Material", getMaterial());
        values.put("MaterialByCustomer", getMaterialByCustomer());
        values.put("PricingDate", getPricingDate());
        values.put("RequestedQuantity", getRequestedQuantity());
        values.put("RequestedQuantityUnit", getRequestedQuantityUnit());
        values.put("RequestedQuantitySAPUnit", getRequestedQuantitySAPUnit());
        values.put("RequestedQuantityISOUnit", getRequestedQuantityISOUnit());
        values.put("ItemGrossWeight", getItemGrossWeight());
        values.put("ItemNetWeight", getItemNetWeight());
        values.put("ItemWeightUnit", getItemWeightUnit());
        values.put("ItemWeightSAPUnit", getItemWeightSAPUnit());
        values.put("ItemWeightISOUnit", getItemWeightISOUnit());
        values.put("ItemVolume", getItemVolume());
        values.put("ItemVolumeUnit", getItemVolumeUnit());
        values.put("ItemVolumeSAPUnit", getItemVolumeSAPUnit());
        values.put("ItemVolumeISOUnit", getItemVolumeISOUnit());
        values.put("OutlineAgreementTargetAmount", getOutlineAgreementTargetAmount());
        values.put("TransactionCurrency", getTransactionCurrency());
        values.put("NetAmount", getNetAmount());
        values.put("MaterialGroup", getMaterialGroup());
        values.put("MaterialPricingGroup", getMaterialPricingGroup());
        values.put("Batch", getBatch());
        values.put("ProductionPlant", getProductionPlant());
        values.put("StorageLocation", getStorageLocation());
        values.put("ShippingPoint", getShippingPoint());
        values.put("IncotermsClassification", getIncotermsClassification());
        values.put("IncotermsTransferLocation", getIncotermsTransferLocation());
        values.put("IncotermsLocation1", getIncotermsLocation1());
        values.put("IncotermsLocation2", getIncotermsLocation2());
        values.put("CustomerPaymentTerms", getCustomerPaymentTerms());
        values.put("SalesDocumentRjcnReason", getSalesDocumentRjcnReason());
        values.put("ItemBillingBlockReason", getItemBillingBlockReason());
        values.put("WBSElement", getWBSElement());
        values.put("ProfitCenter", getProfitCenter());
        values.put("ReferenceSDDocument", getReferenceSDDocument());
        values.put("ReferenceSDDocumentItem", getReferenceSDDocumentItem());
        values.put("SDProcessStatus", getSDProcessStatus());
        values.put("SalesContractValidityStartDate", getSalesContractValidityStartDate());
        values.put("SalesContractValidityEndDate", getSalesContractValidityEndDate());
        values.put("NmbrOfSalesContractValdtyPerd", getNmbrOfSalesContractValdtyPerd());
        values.put("SalesContractValidityPerdUnit", getSalesContractValidityPerdUnit());
        values.put("SalesContractValidityPerdCat", getSalesContractValidityPerdCat());
        values.put("SalesContractSignedDate", getSalesContractSignedDate());
        values.put("SalesContractFollowUpAction", getSalesContractFollowUpAction());
        values.put("SlsContractFollowUpActionDate", getSlsContractFollowUpActionDate());
        values.put("SlsContrItemReleasedQuantity", getSlsContrItemReleasedQuantity());
        values.put("SlsContrItmReldQuantityUnit", getSlsContrItmReldQuantityUnit());
        return values;
    }

    @Override
    protected void fromMap( final Map<String, Object> inputValues )
    {
        final Map<String, Object> values = Maps.newHashMap(inputValues);
        // simple properties
        {
            if( values.containsKey("SalesContract") ) {
                final Object value = values.remove("SalesContract");
                if( (value == null) || (!value.equals(getSalesContract())) ) {
                    setSalesContract(((String) value));
                }
            }
            if( values.containsKey("SalesContractItem") ) {
                final Object value = values.remove("SalesContractItem");
                if( (value == null) || (!value.equals(getSalesContractItem())) ) {
                    setSalesContractItem(((String) value));
                }
            }
            if( values.containsKey("HigherLevelItem") ) {
                final Object value = values.remove("HigherLevelItem");
                if( (value == null) || (!value.equals(getHigherLevelItem())) ) {
                    setHigherLevelItem(((String) value));
                }
            }
            if( values.containsKey("SalesContractItemCategory") ) {
                final Object value = values.remove("SalesContractItemCategory");
                if( (value == null) || (!value.equals(getSalesContractItemCategory())) ) {
                    setSalesContractItemCategory(((String) value));
                }
            }
            if( values.containsKey("SalesContractItemText") ) {
                final Object value = values.remove("SalesContractItemText");
                if( (value == null) || (!value.equals(getSalesContractItemText())) ) {
                    setSalesContractItemText(((String) value));
                }
            }
            if( values.containsKey("PurchaseOrderByCustomer") ) {
                final Object value = values.remove("PurchaseOrderByCustomer");
                if( (value == null) || (!value.equals(getPurchaseOrderByCustomer())) ) {
                    setPurchaseOrderByCustomer(((String) value));
                }
            }
            if( values.containsKey("Material") ) {
                final Object value = values.remove("Material");
                if( (value == null) || (!value.equals(getMaterial())) ) {
                    setMaterial(((String) value));
                }
            }
            if( values.containsKey("MaterialByCustomer") ) {
                final Object value = values.remove("MaterialByCustomer");
                if( (value == null) || (!value.equals(getMaterialByCustomer())) ) {
                    setMaterialByCustomer(((String) value));
                }
            }
            if( values.containsKey("PricingDate") ) {
                final Object value = values.remove("PricingDate");
                if( (value == null) || (!value.equals(getPricingDate())) ) {
                    setPricingDate(((LocalDateTime) value));
                }
            }
            if( values.containsKey("RequestedQuantity") ) {
                final Object value = values.remove("RequestedQuantity");
                if( (value == null) || (!value.equals(getRequestedQuantity())) ) {
                    setRequestedQuantity(((BigDecimal) value));
                }
            }
            if( values.containsKey("RequestedQuantityUnit") ) {
                final Object value = values.remove("RequestedQuantityUnit");
                if( (value == null) || (!value.equals(getRequestedQuantityUnit())) ) {
                    setRequestedQuantityUnit(((String) value));
                }
            }
            if( values.containsKey("RequestedQuantitySAPUnit") ) {
                final Object value = values.remove("RequestedQuantitySAPUnit");
                if( (value == null) || (!value.equals(getRequestedQuantitySAPUnit())) ) {
                    setRequestedQuantitySAPUnit(((String) value));
                }
            }
            if( values.containsKey("RequestedQuantityISOUnit") ) {
                final Object value = values.remove("RequestedQuantityISOUnit");
                if( (value == null) || (!value.equals(getRequestedQuantityISOUnit())) ) {
                    setRequestedQuantityISOUnit(((String) value));
                }
            }
            if( values.containsKey("ItemGrossWeight") ) {
                final Object value = values.remove("ItemGrossWeight");
                if( (value == null) || (!value.equals(getItemGrossWeight())) ) {
                    setItemGrossWeight(((BigDecimal) value));
                }
            }
            if( values.containsKey("ItemNetWeight") ) {
                final Object value = values.remove("ItemNetWeight");
                if( (value == null) || (!value.equals(getItemNetWeight())) ) {
                    setItemNetWeight(((BigDecimal) value));
                }
            }
            if( values.containsKey("ItemWeightUnit") ) {
                final Object value = values.remove("ItemWeightUnit");
                if( (value == null) || (!value.equals(getItemWeightUnit())) ) {
                    setItemWeightUnit(((String) value));
                }
            }
            if( values.containsKey("ItemWeightSAPUnit") ) {
                final Object value = values.remove("ItemWeightSAPUnit");
                if( (value == null) || (!value.equals(getItemWeightSAPUnit())) ) {
                    setItemWeightSAPUnit(((String) value));
                }
            }
            if( values.containsKey("ItemWeightISOUnit") ) {
                final Object value = values.remove("ItemWeightISOUnit");
                if( (value == null) || (!value.equals(getItemWeightISOUnit())) ) {
                    setItemWeightISOUnit(((String) value));
                }
            }
            if( values.containsKey("ItemVolume") ) {
                final Object value = values.remove("ItemVolume");
                if( (value == null) || (!value.equals(getItemVolume())) ) {
                    setItemVolume(((BigDecimal) value));
                }
            }
            if( values.containsKey("ItemVolumeUnit") ) {
                final Object value = values.remove("ItemVolumeUnit");
                if( (value == null) || (!value.equals(getItemVolumeUnit())) ) {
                    setItemVolumeUnit(((String) value));
                }
            }
            if( values.containsKey("ItemVolumeSAPUnit") ) {
                final Object value = values.remove("ItemVolumeSAPUnit");
                if( (value == null) || (!value.equals(getItemVolumeSAPUnit())) ) {
                    setItemVolumeSAPUnit(((String) value));
                }
            }
            if( values.containsKey("ItemVolumeISOUnit") ) {
                final Object value = values.remove("ItemVolumeISOUnit");
                if( (value == null) || (!value.equals(getItemVolumeISOUnit())) ) {
                    setItemVolumeISOUnit(((String) value));
                }
            }
            if( values.containsKey("OutlineAgreementTargetAmount") ) {
                final Object value = values.remove("OutlineAgreementTargetAmount");
                if( (value == null) || (!value.equals(getOutlineAgreementTargetAmount())) ) {
                    setOutlineAgreementTargetAmount(((BigDecimal) value));
                }
            }
            if( values.containsKey("TransactionCurrency") ) {
                final Object value = values.remove("TransactionCurrency");
                if( (value == null) || (!value.equals(getTransactionCurrency())) ) {
                    setTransactionCurrency(((String) value));
                }
            }
            if( values.containsKey("NetAmount") ) {
                final Object value = values.remove("NetAmount");
                if( (value == null) || (!value.equals(getNetAmount())) ) {
                    setNetAmount(((BigDecimal) value));
                }
            }
            if( values.containsKey("MaterialGroup") ) {
                final Object value = values.remove("MaterialGroup");
                if( (value == null) || (!value.equals(getMaterialGroup())) ) {
                    setMaterialGroup(((String) value));
                }
            }
            if( values.containsKey("MaterialPricingGroup") ) {
                final Object value = values.remove("MaterialPricingGroup");
                if( (value == null) || (!value.equals(getMaterialPricingGroup())) ) {
                    setMaterialPricingGroup(((String) value));
                }
            }
            if( values.containsKey("Batch") ) {
                final Object value = values.remove("Batch");
                if( (value == null) || (!value.equals(getBatch())) ) {
                    setBatch(((String) value));
                }
            }
            if( values.containsKey("ProductionPlant") ) {
                final Object value = values.remove("ProductionPlant");
                if( (value == null) || (!value.equals(getProductionPlant())) ) {
                    setProductionPlant(((String) value));
                }
            }
            if( values.containsKey("StorageLocation") ) {
                final Object value = values.remove("StorageLocation");
                if( (value == null) || (!value.equals(getStorageLocation())) ) {
                    setStorageLocation(((String) value));
                }
            }
            if( values.containsKey("ShippingPoint") ) {
                final Object value = values.remove("ShippingPoint");
                if( (value == null) || (!value.equals(getShippingPoint())) ) {
                    setShippingPoint(((String) value));
                }
            }
            if( values.containsKey("IncotermsClassification") ) {
                final Object value = values.remove("IncotermsClassification");
                if( (value == null) || (!value.equals(getIncotermsClassification())) ) {
                    setIncotermsClassification(((String) value));
                }
            }
            if( values.containsKey("IncotermsTransferLocation") ) {
                final Object value = values.remove("IncotermsTransferLocation");
                if( (value == null) || (!value.equals(getIncotermsTransferLocation())) ) {
                    setIncotermsTransferLocation(((String) value));
                }
            }
            if( values.containsKey("IncotermsLocation1") ) {
                final Object value = values.remove("IncotermsLocation1");
                if( (value == null) || (!value.equals(getIncotermsLocation1())) ) {
                    setIncotermsLocation1(((String) value));
                }
            }
            if( values.containsKey("IncotermsLocation2") ) {
                final Object value = values.remove("IncotermsLocation2");
                if( (value == null) || (!value.equals(getIncotermsLocation2())) ) {
                    setIncotermsLocation2(((String) value));
                }
            }
            if( values.containsKey("CustomerPaymentTerms") ) {
                final Object value = values.remove("CustomerPaymentTerms");
                if( (value == null) || (!value.equals(getCustomerPaymentTerms())) ) {
                    setCustomerPaymentTerms(((String) value));
                }
            }
            if( values.containsKey("SalesDocumentRjcnReason") ) {
                final Object value = values.remove("SalesDocumentRjcnReason");
                if( (value == null) || (!value.equals(getSalesDocumentRjcnReason())) ) {
                    setSalesDocumentRjcnReason(((String) value));
                }
            }
            if( values.containsKey("ItemBillingBlockReason") ) {
                final Object value = values.remove("ItemBillingBlockReason");
                if( (value == null) || (!value.equals(getItemBillingBlockReason())) ) {
                    setItemBillingBlockReason(((String) value));
                }
            }
            if( values.containsKey("WBSElement") ) {
                final Object value = values.remove("WBSElement");
                if( (value == null) || (!value.equals(getWBSElement())) ) {
                    setWBSElement(((String) value));
                }
            }
            if( values.containsKey("ProfitCenter") ) {
                final Object value = values.remove("ProfitCenter");
                if( (value == null) || (!value.equals(getProfitCenter())) ) {
                    setProfitCenter(((String) value));
                }
            }
            if( values.containsKey("ReferenceSDDocument") ) {
                final Object value = values.remove("ReferenceSDDocument");
                if( (value == null) || (!value.equals(getReferenceSDDocument())) ) {
                    setReferenceSDDocument(((String) value));
                }
            }
            if( values.containsKey("ReferenceSDDocumentItem") ) {
                final Object value = values.remove("ReferenceSDDocumentItem");
                if( (value == null) || (!value.equals(getReferenceSDDocumentItem())) ) {
                    setReferenceSDDocumentItem(((String) value));
                }
            }
            if( values.containsKey("SDProcessStatus") ) {
                final Object value = values.remove("SDProcessStatus");
                if( (value == null) || (!value.equals(getSDProcessStatus())) ) {
                    setSDProcessStatus(((String) value));
                }
            }
            if( values.containsKey("SalesContractValidityStartDate") ) {
                final Object value = values.remove("SalesContractValidityStartDate");
                if( (value == null) || (!value.equals(getSalesContractValidityStartDate())) ) {
                    setSalesContractValidityStartDate(((LocalDateTime) value));
                }
            }
            if( values.containsKey("SalesContractValidityEndDate") ) {
                final Object value = values.remove("SalesContractValidityEndDate");
                if( (value == null) || (!value.equals(getSalesContractValidityEndDate())) ) {
                    setSalesContractValidityEndDate(((LocalDateTime) value));
                }
            }
            if( values.containsKey("NmbrOfSalesContractValdtyPerd") ) {
                final Object value = values.remove("NmbrOfSalesContractValdtyPerd");
                if( (value == null) || (!value.equals(getNmbrOfSalesContractValdtyPerd())) ) {
                    setNmbrOfSalesContractValdtyPerd(((String) value));
                }
            }
            if( values.containsKey("SalesContractValidityPerdUnit") ) {
                final Object value = values.remove("SalesContractValidityPerdUnit");
                if( (value == null) || (!value.equals(getSalesContractValidityPerdUnit())) ) {
                    setSalesContractValidityPerdUnit(((String) value));
                }
            }
            if( values.containsKey("SalesContractValidityPerdCat") ) {
                final Object value = values.remove("SalesContractValidityPerdCat");
                if( (value == null) || (!value.equals(getSalesContractValidityPerdCat())) ) {
                    setSalesContractValidityPerdCat(((String) value));
                }
            }
            if( values.containsKey("SalesContractSignedDate") ) {
                final Object value = values.remove("SalesContractSignedDate");
                if( (value == null) || (!value.equals(getSalesContractSignedDate())) ) {
                    setSalesContractSignedDate(((LocalDateTime) value));
                }
            }
            if( values.containsKey("SalesContractFollowUpAction") ) {
                final Object value = values.remove("SalesContractFollowUpAction");
                if( (value == null) || (!value.equals(getSalesContractFollowUpAction())) ) {
                    setSalesContractFollowUpAction(((String) value));
                }
            }
            if( values.containsKey("SlsContractFollowUpActionDate") ) {
                final Object value = values.remove("SlsContractFollowUpActionDate");
                if( (value == null) || (!value.equals(getSlsContractFollowUpActionDate())) ) {
                    setSlsContractFollowUpActionDate(((LocalDateTime) value));
                }
            }
            if( values.containsKey("SlsContrItemReleasedQuantity") ) {
                final Object value = values.remove("SlsContrItemReleasedQuantity");
                if( (value == null) || (!value.equals(getSlsContrItemReleasedQuantity())) ) {
                    setSlsContrItemReleasedQuantity(((BigDecimal) value));
                }
            }
            if( values.containsKey("SlsContrItmReldQuantityUnit") ) {
                final Object value = values.remove("SlsContrItmReldQuantityUnit");
                if( (value == null) || (!value.equals(getSlsContrItmReldQuantityUnit())) ) {
                    setSlsContrItmReldQuantityUnit(((String) value));
                }
            }
        }
        // structured properties
        {
        }
        // navigation properties
        {
            if( (values).containsKey("to_Partner") ) {
                final Object value = (values).remove("to_Partner");
                if( value instanceof Iterable ) {
                    if( toPartner == null ) {
                        toPartner = Lists.newArrayList();
                    } else {
                        toPartner = Lists.newArrayList(toPartner);
                    }
                    int i = 0;
                    for( Object item : ((Iterable<?>) value) ) {
                        if( !(item instanceof Map) ) {
                            continue;
                        }
                        SalesContractItemPartner entity;
                        if( toPartner.size() > i ) {
                            entity = toPartner.get(i);
                        } else {
                            entity = new SalesContractItemPartner();
                            toPartner.add(entity);
                        }
                        i = (i + 1);
                        @SuppressWarnings( "unchecked" )
                        final Map<String, Object> inputMap = ((Map<String, Object>) item);
                        entity.fromMap(inputMap);
                    }
                }
            }
            if( (values).containsKey("to_PricingElement") ) {
                final Object value = (values).remove("to_PricingElement");
                if( value instanceof Iterable ) {
                    if( toPricingElement == null ) {
                        toPricingElement = Lists.newArrayList();
                    } else {
                        toPricingElement = Lists.newArrayList(toPricingElement);
                    }
                    int i = 0;
                    for( Object item : ((Iterable<?>) value) ) {
                        if( !(item instanceof Map) ) {
                            continue;
                        }
                        SalesContractItemPrcgElmnt entity;
                        if( toPricingElement.size() > i ) {
                            entity = toPricingElement.get(i);
                        } else {
                            entity = new SalesContractItemPrcgElmnt();
                            toPricingElement.add(entity);
                        }
                        i = (i + 1);
                        @SuppressWarnings( "unchecked" )
                        final Map<String, Object> inputMap = ((Map<String, Object>) item);
                        entity.fromMap(inputMap);
                    }
                }
            }
            if( (values).containsKey("to_SalesContract") ) {
                final Object value = (values).remove("to_SalesContract");
                if( value instanceof Map ) {
                    if( toSalesContract == null ) {
                        toSalesContract = new SalesContract();
                    }
                    @SuppressWarnings( "unchecked" )
                    final Map<String, Object> inputMap = ((Map<String, Object>) value);
                    toSalesContract.fromMap(inputMap);
                }
            }
            if( (values).containsKey("to_Text") ) {
                final Object value = (values).remove("to_Text");
                if( value instanceof Iterable ) {
                    if( toText == null ) {
                        toText = Lists.newArrayList();
                    } else {
                        toText = Lists.newArrayList(toText);
                    }
                    int i = 0;
                    for( Object item : ((Iterable<?>) value) ) {
                        if( !(item instanceof Map) ) {
                            continue;
                        }
                        SalesContractItemText entity;
                        if( toText.size() > i ) {
                            entity = toText.get(i);
                        } else {
                            entity = new SalesContractItemText();
                            toText.add(entity);
                        }
                        i = (i + 1);
                        @SuppressWarnings( "unchecked" )
                        final Map<String, Object> inputMap = ((Map<String, Object>) item);
                        entity.fromMap(inputMap);
                    }
                }
            }
        }
        super.fromMap(values);
    }

    /**
     * Use with available fluent helpers to apply an extension field to query operations.
     *
     * @param fieldName
     *            The name of the extension field as returned by the OData service.
     * @param <T>
     *            The type of the extension field when performing value comparisons.
     * @param fieldType
     *            The Java type to use for the extension field when performing value comparisons.
     * @return A representation of an extension field from this entity.
     */
    @Nonnull
    public static <
        T> SalesContractItemField<T> field( @Nonnull final String fieldName, @Nonnull final Class<T> fieldType )
    {
        return new SalesContractItemField<T>(fieldName);
    }

    /**
     * Use with available fluent helpers to apply an extension field to query operations.
     *
     * @param typeConverter
     *            A TypeConverter<T, DomainT> instance whose first generic type matches the Java type of the field
     * @param fieldName
     *            The name of the extension field as returned by the OData service.
     * @param <T>
     *            The type of the extension field when performing value comparisons.
     * @param <DomainT>
     *            The type of the extension field as returned by the OData service.
     * @return A representation of an extension field from this entity, holding a reference to the given TypeConverter.
     */
    @Nonnull
    public static <T, DomainT> SalesContractItemField<T> field(
        @Nonnull final String fieldName,
        @Nonnull final TypeConverter<T, DomainT> typeConverter )
    {
        return new SalesContractItemField<T>(fieldName, typeConverter);
    }

    @Override
    @Nullable
    public HttpDestinationProperties getDestinationForFetch()
    {
        return super.getDestinationForFetch();
    }

    @Override
    protected void setServicePathForFetch( @Nullable final String servicePathForFetch )
    {
        super.setServicePathForFetch(servicePathForFetch);
    }

    @Override
    public
        void
        attachToService( @Nullable final String servicePath, @Nonnull final HttpDestinationProperties destination )
    {
        super.attachToService(servicePath, destination);
    }

    @Override
    @SuppressWarnings( "deprecation" )
    protected String getDefaultServicePath()
    {
        return (com.sap.cloud.sdk.s4hana.datamodel.odata.services.SalesContractService.DEFAULT_SERVICE_PATH);
    }

    @Nonnull
    @Override
    protected Map<String, Object> toMapOfNavigationProperties()
    {
        final Map<String, Object> values = super.toMapOfNavigationProperties();
        if( toPartner != null ) {
            (values).put("to_Partner", toPartner);
        }
        if( toPricingElement != null ) {
            (values).put("to_PricingElement", toPricingElement);
        }
        if( toSalesContract != null ) {
            (values).put("to_SalesContract", toSalesContract);
        }
        if( toText != null ) {
            (values).put("to_Text", toText);
        }
        return values;
    }

    /**
     * Fetches the <b>SalesContractItemPartner</b> entities (one to many) associated with this entity. This corresponds
     * to the OData navigation property <b>to_Partner</b>.
     * <p>
     * Please note: This method will not cache or persist the query results.
     *
     * @return List containing one or more associated <b>SalesContractItemPartner</b> entities. If no entities are
     *         associated then an empty list is returned.
     * @throws ODataException
     *             If the entity is unmanaged, i.e. it has not been retrieved using the OData VDM's services and
     *             therefore has no ERP configuration context assigned. An entity is managed if it has been either
     *             retrieved using the VDM's services or returned from the VDM's services as the result of a CREATE or
     *             UPDATE call.
     */
    @Nonnull
    public List<SalesContractItemPartner> fetchPartner()
    {
        return fetchFieldAsList("to_Partner", SalesContractItemPartner.class);
    }

    /**
     * Retrieval of associated <b>SalesContractItemPartner</b> entities (one to many). This corresponds to the OData
     * navigation property <b>to_Partner</b>.
     * <p>
     * If the navigation property <b>to_Partner</b> of a queried <b>SalesContractItem</b> is operated lazily, an
     * <b>ODataException</b> can be thrown in case of an OData query error.
     * <p>
     * Please note: <i>Lazy</i> loading of OData entity associations is the process of asynchronous retrieval and
     * persisting of items from a navigation property. If a <i>lazy</i> property is requested by the application for the
     * first time and it has not yet been loaded, an OData query will be run in order to load the missing information
     * and its result will get cached for future invocations.
     *
     * @return List of associated <b>SalesContractItemPartner</b> entities.
     * @throws ODataException
     *             If the entity is unmanaged, i.e. it has not been retrieved using the OData VDM's services and
     *             therefore has no ERP configuration context assigned. An entity is managed if it has been either
     *             retrieved using the VDM's services or returned from the VDM's services as the result of a CREATE or
     *             UPDATE call.
     */
    @Nonnull
    public List<SalesContractItemPartner> getPartnerOrFetch()
    {
        if( toPartner == null ) {
            toPartner = fetchPartner();
        }
        return toPartner;
    }

    /**
     * Retrieval of associated <b>SalesContractItemPartner</b> entities (one to many). This corresponds to the OData
     * navigation property <b>to_Partner</b>.
     * <p>
     * If the navigation property for an entity <b>SalesContractItem</b> has not been resolved yet, this method will
     * <b>not query</b> further information. Instead its <code>Option</code> result state will be <code>empty</code>.
     *
     * @return If the information for navigation property <b>to_Partner</b> is already loaded, the result will contain
     *         the <b>SalesContractItemPartner</b> entities. If not, an <code>Option</code> with result state
     *         <code>empty</code> is returned.
     */
    @Nonnull
    public Option<List<SalesContractItemPartner>> getPartnerIfPresent()
    {
        return Option.of(toPartner);
    }

    /**
     * Overwrites the list of associated <b>SalesContractItemPartner</b> entities for the loaded navigation property
     * <b>to_Partner</b>.
     * <p>
     * If the navigation property <b>to_Partner</b> of a queried <b>SalesContractItem</b> is operated lazily, an
     * <b>ODataException</b> can be thrown in case of an OData query error.
     * <p>
     * Please note: <i>Lazy</i> loading of OData entity associations is the process of asynchronous retrieval and
     * persisting of items from a navigation property. If a <i>lazy</i> property is requested by the application for the
     * first time and it has not yet been loaded, an OData query will be run in order to load the missing information
     * and its result will get cached for future invocations.
     *
     * @param value
     *            List of <b>SalesContractItemPartner</b> entities.
     */
    public void setPartner( @Nonnull final List<SalesContractItemPartner> value )
    {
        if( toPartner == null ) {
            toPartner = Lists.newArrayList();
        }
        toPartner.clear();
        toPartner.addAll(value);
    }

    /**
     * Adds elements to the list of associated <b>SalesContractItemPartner</b> entities. This corresponds to the OData
     * navigation property <b>to_Partner</b>.
     * <p>
     * If the navigation property <b>to_Partner</b> of a queried <b>SalesContractItem</b> is operated lazily, an
     * <b>ODataException</b> can be thrown in case of an OData query error.
     * <p>
     * Please note: <i>Lazy</i> loading of OData entity associations is the process of asynchronous retrieval and
     * persisting of items from a navigation property. If a <i>lazy</i> property is requested by the application for the
     * first time and it has not yet been loaded, an OData query will be run in order to load the missing information
     * and its result will get cached for future invocations.
     *
     * @param entity
     *            Array of <b>SalesContractItemPartner</b> entities.
     */
    public void addPartner( SalesContractItemPartner... entity )
    {
        if( toPartner == null ) {
            toPartner = Lists.newArrayList();
        }
        toPartner.addAll(Lists.newArrayList(entity));
    }

    /**
     * Fetches the <b>SalesContractItemPrcgElmnt</b> entities (one to many) associated with this entity. This
     * corresponds to the OData navigation property <b>to_PricingElement</b>.
     * <p>
     * Please note: This method will not cache or persist the query results.
     *
     * @return List containing one or more associated <b>SalesContractItemPrcgElmnt</b> entities. If no entities are
     *         associated then an empty list is returned.
     * @throws ODataException
     *             If the entity is unmanaged, i.e. it has not been retrieved using the OData VDM's services and
     *             therefore has no ERP configuration context assigned. An entity is managed if it has been either
     *             retrieved using the VDM's services or returned from the VDM's services as the result of a CREATE or
     *             UPDATE call.
     */
    @Nonnull
    public List<SalesContractItemPrcgElmnt> fetchPricingElement()
    {
        return fetchFieldAsList("to_PricingElement", SalesContractItemPrcgElmnt.class);
    }

    /**
     * Retrieval of associated <b>SalesContractItemPrcgElmnt</b> entities (one to many). This corresponds to the OData
     * navigation property <b>to_PricingElement</b>.
     * <p>
     * If the navigation property <b>to_PricingElement</b> of a queried <b>SalesContractItem</b> is operated lazily, an
     * <b>ODataException</b> can be thrown in case of an OData query error.
     * <p>
     * Please note: <i>Lazy</i> loading of OData entity associations is the process of asynchronous retrieval and
     * persisting of items from a navigation property. If a <i>lazy</i> property is requested by the application for the
     * first time and it has not yet been loaded, an OData query will be run in order to load the missing information
     * and its result will get cached for future invocations.
     *
     * @return List of associated <b>SalesContractItemPrcgElmnt</b> entities.
     * @throws ODataException
     *             If the entity is unmanaged, i.e. it has not been retrieved using the OData VDM's services and
     *             therefore has no ERP configuration context assigned. An entity is managed if it has been either
     *             retrieved using the VDM's services or returned from the VDM's services as the result of a CREATE or
     *             UPDATE call.
     */
    @Nonnull
    public List<SalesContractItemPrcgElmnt> getPricingElementOrFetch()
    {
        if( toPricingElement == null ) {
            toPricingElement = fetchPricingElement();
        }
        return toPricingElement;
    }

    /**
     * Retrieval of associated <b>SalesContractItemPrcgElmnt</b> entities (one to many). This corresponds to the OData
     * navigation property <b>to_PricingElement</b>.
     * <p>
     * If the navigation property for an entity <b>SalesContractItem</b> has not been resolved yet, this method will
     * <b>not query</b> further information. Instead its <code>Option</code> result state will be <code>empty</code>.
     *
     * @return If the information for navigation property <b>to_PricingElement</b> is already loaded, the result will
     *         contain the <b>SalesContractItemPrcgElmnt</b> entities. If not, an <code>Option</code> with result state
     *         <code>empty</code> is returned.
     */
    @Nonnull
    public Option<List<SalesContractItemPrcgElmnt>> getPricingElementIfPresent()
    {
        return Option.of(toPricingElement);
    }

    /**
     * Overwrites the list of associated <b>SalesContractItemPrcgElmnt</b> entities for the loaded navigation property
     * <b>to_PricingElement</b>.
     * <p>
     * If the navigation property <b>to_PricingElement</b> of a queried <b>SalesContractItem</b> is operated lazily, an
     * <b>ODataException</b> can be thrown in case of an OData query error.
     * <p>
     * Please note: <i>Lazy</i> loading of OData entity associations is the process of asynchronous retrieval and
     * persisting of items from a navigation property. If a <i>lazy</i> property is requested by the application for the
     * first time and it has not yet been loaded, an OData query will be run in order to load the missing information
     * and its result will get cached for future invocations.
     *
     * @param value
     *            List of <b>SalesContractItemPrcgElmnt</b> entities.
     */
    public void setPricingElement( @Nonnull final List<SalesContractItemPrcgElmnt> value )
    {
        if( toPricingElement == null ) {
            toPricingElement = Lists.newArrayList();
        }
        toPricingElement.clear();
        toPricingElement.addAll(value);
    }

    /**
     * Adds elements to the list of associated <b>SalesContractItemPrcgElmnt</b> entities. This corresponds to the OData
     * navigation property <b>to_PricingElement</b>.
     * <p>
     * If the navigation property <b>to_PricingElement</b> of a queried <b>SalesContractItem</b> is operated lazily, an
     * <b>ODataException</b> can be thrown in case of an OData query error.
     * <p>
     * Please note: <i>Lazy</i> loading of OData entity associations is the process of asynchronous retrieval and
     * persisting of items from a navigation property. If a <i>lazy</i> property is requested by the application for the
     * first time and it has not yet been loaded, an OData query will be run in order to load the missing information
     * and its result will get cached for future invocations.
     *
     * @param entity
     *            Array of <b>SalesContractItemPrcgElmnt</b> entities.
     */
    public void addPricingElement( SalesContractItemPrcgElmnt... entity )
    {
        if( toPricingElement == null ) {
            toPricingElement = Lists.newArrayList();
        }
        toPricingElement.addAll(Lists.newArrayList(entity));
    }

    /**
     * Fetches the <b>SalesContract</b> entity (one to one) associated with this entity. This corresponds to the OData
     * navigation property <b>to_SalesContract</b>.
     * <p>
     * Please note: This method will not cache or persist the query results.
     *
     * @return The single associated <b>SalesContract</b> entity, or {@code null} if an entity is not associated.
     * @throws ODataException
     *             If the entity is unmanaged, i.e. it has not been retrieved using the OData VDM's services and
     *             therefore has no ERP configuration context assigned. An entity is managed if it has been either
     *             retrieved using the VDM's services or returned from the VDM's services as the result of a CREATE or
     *             UPDATE call.
     */
    @Nullable
    public SalesContract fetchSalesContract()
    {
        return fetchFieldAsSingle("to_SalesContract", SalesContract.class);
    }

    /**
     * Retrieval of associated <b>SalesContract</b> entity (one to one). This corresponds to the OData navigation
     * property <b>to_SalesContract</b>.
     * <p>
     * If the navigation property <b>to_SalesContract</b> of a queried <b>SalesContractItem</b> is operated lazily, an
     * <b>ODataException</b> can be thrown in case of an OData query error.
     * <p>
     * Please note: <i>Lazy</i> loading of OData entity associations is the process of asynchronous retrieval and
     * persisting of items from a navigation property. If a <i>lazy</i> property is requested by the application for the
     * first time and it has not yet been loaded, an OData query will be run in order to load the missing information
     * and its result will get cached for future invocations.
     *
     * @return List of associated <b>SalesContract</b> entity.
     * @throws ODataException
     *             If the entity is unmanaged, i.e. it has not been retrieved using the OData VDM's services and
     *             therefore has no ERP configuration context assigned. An entity is managed if it has been either
     *             retrieved using the VDM's services or returned from the VDM's services as the result of a CREATE or
     *             UPDATE call.
     */
    @Nullable
    public SalesContract getSalesContractOrFetch()
    {
        if( toSalesContract == null ) {
            toSalesContract = fetchSalesContract();
        }
        return toSalesContract;
    }

    /**
     * Retrieval of associated <b>SalesContract</b> entity (one to one). This corresponds to the OData navigation
     * property <b>to_SalesContract</b>.
     * <p>
     * If the navigation property for an entity <b>SalesContractItem</b> has not been resolved yet, this method will
     * <b>not query</b> further information. Instead its <code>Option</code> result state will be <code>empty</code>.
     *
     * @return If the information for navigation property <b>to_SalesContract</b> is already loaded, the result will
     *         contain the <b>SalesContract</b> entity. If not, an <code>Option</code> with result state
     *         <code>empty</code> is returned.
     */
    @Nonnull
    public Option<SalesContract> getSalesContractIfPresent()
    {
        return Option.of(toSalesContract);
    }

    /**
     * Overwrites the associated <b>SalesContract</b> entity for the loaded navigation property <b>to_SalesContract</b>.
     *
     * @param value
     *            New <b>SalesContract</b> entity.
     */
    public void setSalesContract( final SalesContract value )
    {
        toSalesContract = value;
    }

    /**
     * Fetches the <b>SalesContractItemText</b> entities (one to many) associated with this entity. This corresponds to
     * the OData navigation property <b>to_Text</b>.
     * <p>
     * Please note: This method will not cache or persist the query results.
     *
     * @return List containing one or more associated <b>SalesContractItemText</b> entities. If no entities are
     *         associated then an empty list is returned.
     * @throws ODataException
     *             If the entity is unmanaged, i.e. it has not been retrieved using the OData VDM's services and
     *             therefore has no ERP configuration context assigned. An entity is managed if it has been either
     *             retrieved using the VDM's services or returned from the VDM's services as the result of a CREATE or
     *             UPDATE call.
     */
    @Nonnull
    public List<SalesContractItemText> fetchText()
    {
        return fetchFieldAsList("to_Text", SalesContractItemText.class);
    }

    /**
     * Retrieval of associated <b>SalesContractItemText</b> entities (one to many). This corresponds to the OData
     * navigation property <b>to_Text</b>.
     * <p>
     * If the navigation property <b>to_Text</b> of a queried <b>SalesContractItem</b> is operated lazily, an
     * <b>ODataException</b> can be thrown in case of an OData query error.
     * <p>
     * Please note: <i>Lazy</i> loading of OData entity associations is the process of asynchronous retrieval and
     * persisting of items from a navigation property. If a <i>lazy</i> property is requested by the application for the
     * first time and it has not yet been loaded, an OData query will be run in order to load the missing information
     * and its result will get cached for future invocations.
     *
     * @return List of associated <b>SalesContractItemText</b> entities.
     * @throws ODataException
     *             If the entity is unmanaged, i.e. it has not been retrieved using the OData VDM's services and
     *             therefore has no ERP configuration context assigned. An entity is managed if it has been either
     *             retrieved using the VDM's services or returned from the VDM's services as the result of a CREATE or
     *             UPDATE call.
     */
    @Nonnull
    public List<SalesContractItemText> getTextOrFetch()
    {
        if( toText == null ) {
            toText = fetchText();
        }
        return toText;
    }

    /**
     * Retrieval of associated <b>SalesContractItemText</b> entities (one to many). This corresponds to the OData
     * navigation property <b>to_Text</b>.
     * <p>
     * If the navigation property for an entity <b>SalesContractItem</b> has not been resolved yet, this method will
     * <b>not query</b> further information. Instead its <code>Option</code> result state will be <code>empty</code>.
     *
     * @return If the information for navigation property <b>to_Text</b> is already loaded, the result will contain the
     *         <b>SalesContractItemText</b> entities. If not, an <code>Option</code> with result state
     *         <code>empty</code> is returned.
     */
    @Nonnull
    public Option<List<SalesContractItemText>> getTextIfPresent()
    {
        return Option.of(toText);
    }

    /**
     * Overwrites the list of associated <b>SalesContractItemText</b> entities for the loaded navigation property
     * <b>to_Text</b>.
     * <p>
     * If the navigation property <b>to_Text</b> of a queried <b>SalesContractItem</b> is operated lazily, an
     * <b>ODataException</b> can be thrown in case of an OData query error.
     * <p>
     * Please note: <i>Lazy</i> loading of OData entity associations is the process of asynchronous retrieval and
     * persisting of items from a navigation property. If a <i>lazy</i> property is requested by the application for the
     * first time and it has not yet been loaded, an OData query will be run in order to load the missing information
     * and its result will get cached for future invocations.
     *
     * @param value
     *            List of <b>SalesContractItemText</b> entities.
     */
    public void setText( @Nonnull final List<SalesContractItemText> value )
    {
        if( toText == null ) {
            toText = Lists.newArrayList();
        }
        toText.clear();
        toText.addAll(value);
    }

    /**
     * Adds elements to the list of associated <b>SalesContractItemText</b> entities. This corresponds to the OData
     * navigation property <b>to_Text</b>.
     * <p>
     * If the navigation property <b>to_Text</b> of a queried <b>SalesContractItem</b> is operated lazily, an
     * <b>ODataException</b> can be thrown in case of an OData query error.
     * <p>
     * Please note: <i>Lazy</i> loading of OData entity associations is the process of asynchronous retrieval and
     * persisting of items from a navigation property. If a <i>lazy</i> property is requested by the application for the
     * first time and it has not yet been loaded, an OData query will be run in order to load the missing information
     * and its result will get cached for future invocations.
     *
     * @param entity
     *            Array of <b>SalesContractItemText</b> entities.
     */
    public void addText( SalesContractItemText... entity )
    {
        if( toText == null ) {
            toText = Lists.newArrayList();
        }
        toText.addAll(Lists.newArrayList(entity));
    }

    /**
     * Helper class to allow for fluent creation of SalesContractItem instances.
     *
     */
    public final static class SalesContractItemBuilder
    {

        private List<SalesContractItemPartner> toPartner = Lists.newArrayList();
        private List<SalesContractItemPrcgElmnt> toPricingElement = Lists.newArrayList();
        private SalesContract toSalesContract;
        private String salesContract = null;
        private List<SalesContractItemText> toText = Lists.newArrayList();

        private SalesContractItem.SalesContractItemBuilder toPartner( final List<SalesContractItemPartner> value )
        {
            toPartner.addAll(value);
            return this;
        }

        /**
         * Navigation property <b>to_Partner</b> for <b>SalesContractItem</b> to multiple
         * <b>SalesContractItemPartner</b>.
         *
         * @param value
         *            The SalesContractItemPartners to build this SalesContractItem with.
         * @return This Builder to allow for a fluent interface.
         */
        @Nonnull
        public SalesContractItem.SalesContractItemBuilder partner( SalesContractItemPartner... value )
        {
            return toPartner(Lists.newArrayList(value));
        }

        private SalesContractItem.SalesContractItemBuilder toPricingElement(
            final List<SalesContractItemPrcgElmnt> value )
        {
            toPricingElement.addAll(value);
            return this;
        }

        /**
         * Navigation property <b>to_PricingElement</b> for <b>SalesContractItem</b> to multiple
         * <b>SalesContractItemPrcgElmnt</b>.
         *
         * @param value
         *            The SalesContractItemPrcgElmnts to build this SalesContractItem with.
         * @return This Builder to allow for a fluent interface.
         */
        @Nonnull
        public SalesContractItem.SalesContractItemBuilder pricingElement( SalesContractItemPrcgElmnt... value )
        {
            return toPricingElement(Lists.newArrayList(value));
        }

        private SalesContractItem.SalesContractItemBuilder toSalesContract( final SalesContract value )
        {
            toSalesContract = value;
            return this;
        }

        /**
         * Navigation property <b>to_SalesContract</b> for <b>SalesContractItem</b> to single <b>SalesContract</b>.
         *
         * @param value
         *            The SalesContract to build this SalesContractItem with.
         * @return This Builder to allow for a fluent interface.
         */
        @Nonnull
        public SalesContractItem.SalesContractItemBuilder salesContract( final SalesContract value )
        {
            return toSalesContract(value);
        }

        /**
         * (Key Field) Constraints: Not nullable, Maximum length: 10
         * <p>
         * Original property name from the Odata EDM: <b>SalesContract</b>
         * </p>
         *
         * @param value
         *            The salesContract to build this SalesContractItem with.
         * @return This Builder to allow for a fluent interface.
         */
        @Nonnull
        public SalesContractItem.SalesContractItemBuilder salesContract( final String value )
        {
            salesContract = value;
            return this;
        }

        private SalesContractItem.SalesContractItemBuilder toText( final List<SalesContractItemText> value )
        {
            toText.addAll(value);
            return this;
        }

        /**
         * Navigation property <b>to_Text</b> for <b>SalesContractItem</b> to multiple <b>SalesContractItemText</b>.
         *
         * @param value
         *            The SalesContractItemTexts to build this SalesContractItem with.
         * @return This Builder to allow for a fluent interface.
         */
        @Nonnull
        public SalesContractItem.SalesContractItemBuilder text( SalesContractItemText... value )
        {
            return toText(Lists.newArrayList(value));
        }

    }

}
