/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.salesinquiry;

import java.util.Map;

import javax.annotation.Nonnull;

import com.google.common.collect.Maps;
import com.sap.cloud.sdk.datamodel.odata.helper.FluentHelperByKey;
import com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.salesinquiry.selectable.SalesInquiryItemSelectable;

/**
 * Fluent helper to fetch a single
 * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.salesinquiry.SalesInquiryItem SalesInquiryItem} entity
 * using key fields. This fluent helper allows methods which modify the underlying query to be called before executing
 * the query itself.
 *
 */
public class SalesInquiryItemByKeyFluentHelper
    extends
    FluentHelperByKey<SalesInquiryItemByKeyFluentHelper, SalesInquiryItem, SalesInquiryItemSelectable>
{

    private final Map<String, Object> key = Maps.newHashMap();

    /**
     * Creates a fluent helper object that will fetch a single
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.salesinquiry.SalesInquiryItem SalesInquiryItem} entity
     * with the provided key field values. To perform execution, call the {@link #executeRequest executeRequest} method
     * on the fluent helper object.
     *
     * @param entityCollection
     *            Entity Collection to be used to fetch a single {@code SalesInquiryItem}
     * @param servicePath
     *            Service path to be used to fetch a single {@code SalesInquiryItem}
     * @param salesInquiry
     *            Sales Inquiry
     *            <p>
     *            Constraints: Not nullable, Maximum length: 10
     *            </p>
     * @param salesInquiryItem
     *            Sales Inquiry Item
     *            <p>
     *            Constraints: Not nullable, Maximum length: 6
     *            </p>
     */
    public SalesInquiryItemByKeyFluentHelper(
        @Nonnull final String servicePath,
        @Nonnull final String entityCollection,
        final String salesInquiry,
        final String salesInquiryItem )
    {
        super(servicePath, entityCollection);
        this.key.put("SalesInquiry", salesInquiry);
        this.key.put("SalesInquiryItem", salesInquiryItem);
    }

    @Override
    @Nonnull
    protected Class<SalesInquiryItem> getEntityClass()
    {
        return SalesInquiryItem.class;
    }

    @Override
    @Nonnull
    protected Map<String, Object> getKey()
    {
        return key;
    }

}
