/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.salesorder;

import java.util.Map;

import javax.annotation.Nonnull;

import com.google.common.collect.Maps;
import com.sap.cloud.sdk.datamodel.odata.helper.FluentHelperByKey;
import com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.salesorder.selectable.SalesOrderItemPartnerAddressSelectable;

/**
 * Fluent helper to fetch a single
 * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.salesorder.SalesOrderItemPartnerAddress
 * SalesOrderItemPartnerAddress} entity using key fields. This fluent helper allows methods which modify the underlying
 * query to be called before executing the query itself.
 *
 */
public class SalesOrderItemPartnerAddressByKeyFluentHelper
    extends
    FluentHelperByKey<SalesOrderItemPartnerAddressByKeyFluentHelper, SalesOrderItemPartnerAddress, SalesOrderItemPartnerAddressSelectable>
{

    private final Map<String, Object> key = Maps.newHashMap();

    /**
     * Creates a fluent helper object that will fetch a single
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.salesorder.SalesOrderItemPartnerAddress
     * SalesOrderItemPartnerAddress} entity with the provided key field values. To perform execution, call the
     * {@link #executeRequest executeRequest} method on the fluent helper object.
     *
     * @param entityCollection
     *            Entity Collection to be used to fetch a single {@code SalesOrderItemPartnerAddress}
     * @param partnerFunction
     *            Partner Function
     *            <p>
     *            Constraints: Not nullable, Maximum length: 2
     *            </p>
     * @param salesOrderItem
     *            Item number of the SD document
     *            <p>
     *            Constraints: Not nullable, Maximum length: 6
     *            </p>
     * @param addressRepresentationCode
     *            Version ID for International Addresses
     *            <p>
     *            Constraints: Not nullable, Maximum length: 1
     *            </p>
     * @param servicePath
     *            Service path to be used to fetch a single {@code SalesOrderItemPartnerAddress}
     * @param salesOrder
     *            Sales and Distribution Document Number
     *            <p>
     *            Constraints: Not nullable, Maximum length: 10
     *            </p>
     */
    public SalesOrderItemPartnerAddressByKeyFluentHelper(
        @Nonnull final String servicePath,
        @Nonnull final String entityCollection,
        final String salesOrder,
        final String salesOrderItem,
        final String partnerFunction,
        final String addressRepresentationCode )
    {
        super(servicePath, entityCollection);
        this.key.put("SalesOrder", salesOrder);
        this.key.put("SalesOrderItem", salesOrderItem);
        this.key.put("PartnerFunction", partnerFunction);
        this.key.put("AddressRepresentationCode", addressRepresentationCode);
    }

    @Override
    @Nonnull
    protected Class<SalesOrderItemPartnerAddress> getEntityClass()
    {
        return SalesOrderItemPartnerAddress.class;
    }

    @Override
    @Nonnull
    protected Map<String, Object> getKey()
    {
        return key;
    }

}
