/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.salesorder;

import java.util.Map;

import javax.annotation.Nonnull;

import com.google.common.collect.Maps;
import com.sap.cloud.sdk.datamodel.odata.helper.FluentHelperByKey;
import com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.salesorder.selectable.SalesOrderScheduleLineSelectable;

/**
 * Fluent helper to fetch a single
 * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.salesorder.SalesOrderScheduleLine SalesOrderScheduleLine}
 * entity using key fields. This fluent helper allows methods which modify the underlying query to be called before
 * executing the query itself.
 *
 */
public class SalesOrderScheduleLineByKeyFluentHelper
    extends
    FluentHelperByKey<SalesOrderScheduleLineByKeyFluentHelper, SalesOrderScheduleLine, SalesOrderScheduleLineSelectable>
{

    private final Map<String, Object> key = Maps.newHashMap();

    /**
     * Creates a fluent helper object that will fetch a single
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.salesorder.SalesOrderScheduleLine
     * SalesOrderScheduleLine} entity with the provided key field values. To perform execution, call the
     * {@link #executeRequest executeRequest} method on the fluent helper object.
     *
     * @param entityCollection
     *            Entity Collection to be used to fetch a single {@code SalesOrderScheduleLine}
     * @param salesOrderItem
     *            Sales Document Item
     *            <p>
     *            Constraints: Not nullable, Maximum length: 6
     *            </p>
     * @param servicePath
     *            Service path to be used to fetch a single {@code SalesOrderScheduleLine}
     * @param scheduleLine
     *            Schedule Line Number
     *            <p>
     *            Constraints: Not nullable, Maximum length: 4
     *            </p>
     * @param salesOrder
     *            Sales Document
     *            <p>
     *            Constraints: Not nullable, Maximum length: 10
     *            </p>
     */
    public SalesOrderScheduleLineByKeyFluentHelper(
        @Nonnull final String servicePath,
        @Nonnull final String entityCollection,
        final String salesOrder,
        final String salesOrderItem,
        final String scheduleLine )
    {
        super(servicePath, entityCollection);
        this.key.put("SalesOrder", salesOrder);
        this.key.put("SalesOrderItem", salesOrderItem);
        this.key.put("ScheduleLine", scheduleLine);
    }

    @Override
    @Nonnull
    protected Class<SalesOrderScheduleLine> getEntityClass()
    {
        return SalesOrderScheduleLine.class;
    }

    @Override
    @Nonnull
    protected Map<String, Object> getKey()
    {
        return key;
    }

}
