/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.salesorder;

import java.util.Map;

import javax.annotation.Nonnull;

import com.google.common.collect.Maps;
import com.sap.cloud.sdk.datamodel.odata.helper.FluentHelperByKey;
import com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.salesorder.selectable.SlsOrderItemBillingPlanItemSelectable;

/**
 * Fluent helper to fetch a single
 * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.salesorder.SlsOrderItemBillingPlanItem
 * SlsOrderItemBillingPlanItem} entity using key fields. This fluent helper allows methods which modify the underlying
 * query to be called before executing the query itself.
 *
 */
public class SlsOrderItemBillingPlanItemByKeyFluentHelper
    extends
    FluentHelperByKey<SlsOrderItemBillingPlanItemByKeyFluentHelper, SlsOrderItemBillingPlanItem, SlsOrderItemBillingPlanItemSelectable>
{

    private final Map<String, Object> key = Maps.newHashMap();

    /**
     * Creates a fluent helper object that will fetch a single
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.salesorder.SlsOrderItemBillingPlanItem
     * SlsOrderItemBillingPlanItem} entity with the provided key field values. To perform execution, call the
     * {@link #executeRequest executeRequest} method on the fluent helper object.
     *
     * @param entityCollection
     *            Entity Collection to be used to fetch a single {@code SlsOrderItemBillingPlanItem}
     * @param salesOrderItem
     *            Sales Order Item
     *            <p>
     *            Constraints: Not nullable, Maximum length: 6
     *            </p>
     * @param servicePath
     *            Service path to be used to fetch a single {@code SlsOrderItemBillingPlanItem}
     * @param billingPlan
     *            Billing Plan Number / Invoicing Plan Number
     *            <p>
     *            Constraints: Not nullable, Maximum length: 10
     *            </p>
     * @param salesOrder
     *            Sales Order
     *            <p>
     *            Constraints: Not nullable, Maximum length: 10
     *            </p>
     * @param billingPlanItem
     *            Item for billing plan/invoice plan/payment cards
     *            <p>
     *            Constraints: Not nullable, Maximum length: 6
     *            </p>
     */
    public SlsOrderItemBillingPlanItemByKeyFluentHelper(
        @Nonnull final String servicePath,
        @Nonnull final String entityCollection,
        final String salesOrder,
        final String salesOrderItem,
        final String billingPlan,
        final String billingPlanItem )
    {
        super(servicePath, entityCollection);
        this.key.put("SalesOrder", salesOrder);
        this.key.put("SalesOrderItem", salesOrderItem);
        this.key.put("BillingPlan", billingPlan);
        this.key.put("BillingPlanItem", billingPlanItem);
    }

    @Override
    @Nonnull
    protected Class<SlsOrderItemBillingPlanItem> getEntityClass()
    {
        return SlsOrderItemBillingPlanItem.class;
    }

    @Override
    @Nonnull
    protected Map<String, Object> getKey()
    {
        return key;
    }

}
