/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.salesquotation;

import java.util.Map;

import javax.annotation.Nonnull;

import com.google.common.collect.Maps;
import com.sap.cloud.sdk.datamodel.odata.helper.FluentHelperByKey;
import com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.salesquotation.selectable.SlsQtanItemRelatedObjectSelectable;

/**
 * Fluent helper to fetch a single
 * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.salesquotation.SlsQtanItemRelatedObject
 * SlsQtanItemRelatedObject} entity using key fields. This fluent helper allows methods which modify the underlying
 * query to be called before executing the query itself.
 *
 */
public class SlsQtanItemRelatedObjectByKeyFluentHelper
    extends
    FluentHelperByKey<SlsQtanItemRelatedObjectByKeyFluentHelper, SlsQtanItemRelatedObject, SlsQtanItemRelatedObjectSelectable>
{

    private final Map<String, Object> key = Maps.newHashMap();

    /**
     * Creates a fluent helper object that will fetch a single
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.salesquotation.SlsQtanItemRelatedObject
     * SlsQtanItemRelatedObject} entity with the provided key field values. To perform execution, call the
     * {@link #executeRequest executeRequest} method on the fluent helper object.
     *
     * @param entityCollection
     *            Entity Collection to be used to fetch a single {@code SlsQtanItemRelatedObject}
     * @param salesQuotation
     *            Sales Quotation
     *            <p>
     *            Constraints: Not nullable, Maximum length: 10
     *            </p>
     * @param salesQuotationItem
     *            Sales Quotation Item
     *            <p>
     *            Constraints: Not nullable, Maximum length: 6
     *            </p>
     * @param servicePath
     *            Service path to be used to fetch a single {@code SlsQtanItemRelatedObject}
     * @param sDDocRelatedObjectSequenceNmbr
     *            Sequence Number of the Related Object of an SD Document
     *            <p>
     *            Constraints: Not nullable, Maximum length: 4
     *            </p>
     */
    public SlsQtanItemRelatedObjectByKeyFluentHelper(
        @Nonnull final String servicePath,
        @Nonnull final String entityCollection,
        final String salesQuotation,
        final String salesQuotationItem,
        final String sDDocRelatedObjectSequenceNmbr )
    {
        super(servicePath, entityCollection);
        this.key.put("SalesQuotation", salesQuotation);
        this.key.put("SalesQuotationItem", salesQuotationItem);
        this.key.put("SDDocRelatedObjectSequenceNmbr", sDDocRelatedObjectSequenceNmbr);
    }

    @Override
    @Nonnull
    protected Class<SlsQtanItemRelatedObject> getEntityClass()
    {
        return SlsQtanItemRelatedObject.class;
    }

    @Override
    @Nonnull
    protected Map<String, Object> getKey()
    {
        return key;
    }

}
