/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.schedulingagreement;

import java.net.URI;
import java.util.Map;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpUriRequest;

import com.google.common.collect.Maps;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.sap.cloud.sdk.cloudplatform.connectivity.HttpDestinationProperties;
import com.sap.cloud.sdk.datamodel.odata.helper.SingleValuedFluentHelperFunction;

/**
 * Fluent helper for the <b>ItemRelease</b> OData function import.
 *
 */
public class ItemReleaseFluentHelper
    extends
    SingleValuedFluentHelperFunction<ItemReleaseFluentHelper, ActionResponse, ActionResponse>
{

    private final Map<String, Object> values = Maps.newHashMap();

    /**
     * Creates a fluent helper object that will execute the <b>ItemRelease</b> OData function import with the provided
     * parameters. To perform execution, call the {@link #executeRequest executeRequest} method on the fluent helper
     * object.
     *
     * @param schedulingAgreement
     *            Scheduling Agreement
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Not nullable, Maximum length: 10
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>SchedulingAgreement</b>
     *            </p>
     * @param servicePath
     *            Service path to be used to call the functions against.
     * @param schedulingAgreementReleaseType
     *            Scheduling Agreement Release Type
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Not nullable, Maximum length: 1
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>SchedulingAgreementReleaseType</b>
     *            </p>
     * @param schedulingAgreementItem
     *            Scheduling Agreement Item
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Not nullable, Maximum length: 5
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>SchedulingAgreementItem</b>
     *            </p>
     */
    public ItemReleaseFluentHelper(
        @Nonnull final String servicePath,
        @Nonnull final String schedulingAgreement,
        @Nonnull final String schedulingAgreementItem,
        @Nonnull final String schedulingAgreementReleaseType )
    {
        super(servicePath);
        values.put("SchedulingAgreement", schedulingAgreement);
        values.put("SchedulingAgreementItem", schedulingAgreementItem);
        values.put("SchedulingAgreementReleaseType", schedulingAgreementReleaseType);
    }

    @Override
    @Nonnull
    protected Class<ActionResponse> getEntityClass()
    {
        return ActionResponse.class;
    }

    @Override
    @Nonnull
    protected String getFunctionName()
    {
        return "ItemRelease";
    }

    @Override
    @Nullable
    protected JsonElement refineJsonResponse( @Nullable JsonElement jsonElement )
    {
        if( (jsonElement instanceof JsonObject) && ((JsonObject) jsonElement).has(getFunctionName()) ) {
            jsonElement = ((JsonObject) jsonElement).get(getFunctionName());
        }
        return super.refineJsonResponse(jsonElement);
    }

    @Override
    @Nonnull
    protected Map<String, Object> getParameters()
    {
        return values;
    }

    @Override
    @Nonnull
    protected HttpUriRequest createRequest( @Nonnull final URI uri )
    {
        return new HttpPost(uri);
    }

    /**
     * Execute this function import.
     *
     */
    @Override
    @Nullable
    public ActionResponse executeRequest( @Nonnull final HttpDestinationProperties destination )
    {
        return super.executeSingle(destination);
    }

}
