/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.segment;

import java.util.Map;

import javax.annotation.Nonnull;

import com.google.common.collect.Maps;
import com.sap.cloud.sdk.datamodel.odata.helper.FluentHelperByKey;
import com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.segment.selectable.SegmentTextSelectable;

/**
 * Fluent helper to fetch a single {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.segment.SegmentText
 * SegmentText} entity using key fields. This fluent helper allows methods which modify the underlying query to be
 * called before executing the query itself.
 *
 */
public class SegmentTextByKeyFluentHelper
    extends
    FluentHelperByKey<SegmentTextByKeyFluentHelper, SegmentText, SegmentTextSelectable>
{

    private final Map<String, Object> key = Maps.newHashMap();

    /**
     * Creates a fluent helper object that will fetch a single
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.segment.SegmentText SegmentText} entity with the
     * provided key field values. To perform execution, call the {@link #executeRequest executeRequest} method on the
     * fluent helper object.
     *
     * @param entityCollection
     *            Entity Collection to be used to fetch a single {@code SegmentText}
     * @param servicePath
     *            Service path to be used to fetch a single {@code SegmentText}
     * @param segment
     *            Segment for Segmental Reporting
     *            <p>
     *            Constraints: Not nullable, Maximum length: 10
     *            </p>
     * @param language
     *            Language Key
     *            <p>
     *            Constraints: Not nullable, Maximum length: 2
     *            </p>
     */
    public SegmentTextByKeyFluentHelper(
        @Nonnull final String servicePath,
        @Nonnull final String entityCollection,
        final String language,
        final String segment )
    {
        super(servicePath, entityCollection);
        this.key.put("Language", language);
        this.key.put("Segment", segment);
    }

    @Override
    @Nonnull
    protected Class<SegmentText> getEntityClass()
    {
        return SegmentText.class;
    }

    @Override
    @Nonnull
    protected Map<String, Object> getKey()
    {
        return key;
    }

}
