/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.servicecontract;

import java.util.Map;

import javax.annotation.Nonnull;

import com.google.common.collect.Maps;
import com.sap.cloud.sdk.datamodel.odata.helper.FluentHelperByKey;
import com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.servicecontract.selectable.ServiceContractItemSelectable;

/**
 * Fluent helper to fetch a single
 * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.servicecontract.ServiceContractItem ServiceContractItem}
 * entity using key fields. This fluent helper allows methods which modify the underlying query to be called before
 * executing the query itself.
 *
 */
public class ServiceContractItemByKeyFluentHelper
    extends
    FluentHelperByKey<ServiceContractItemByKeyFluentHelper, ServiceContractItem, ServiceContractItemSelectable>
{

    private final Map<String, Object> key = Maps.newHashMap();

    /**
     * Creates a fluent helper object that will fetch a single
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.servicecontract.ServiceContractItem
     * ServiceContractItem} entity with the provided key field values. To perform execution, call the
     * {@link #executeRequest executeRequest} method on the fluent helper object.
     *
     * @param entityCollection
     *            Entity Collection to be used to fetch a single {@code ServiceContractItem}
     * @param serviceContractItem
     *            Shortened Item Number in Document
     *            <p>
     *            Constraints: Not nullable, Maximum length: 6
     *            </p>
     * @param servicePath
     *            Service path to be used to fetch a single {@code ServiceContractItem}
     * @param serviceContract
     *            Transaction ID
     *            <p>
     *            Constraints: Not nullable, Maximum length: 10
     *            </p>
     */
    public ServiceContractItemByKeyFluentHelper(
        @Nonnull final String servicePath,
        @Nonnull final String entityCollection,
        final String serviceContract,
        final String serviceContractItem )
    {
        super(servicePath, entityCollection);
        this.key.put("ServiceContract", serviceContract);
        this.key.put("ServiceContractItem", serviceContractItem);
    }

    @Override
    @Nonnull
    protected Class<ServiceContractItem> getEntityClass()
    {
        return ServiceContractItem.class;
    }

    @Override
    @Nonnull
    protected Map<String, Object> getKey()
    {
        return key;
    }

}
