/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.supplierinvoiceintegration;

import java.net.URI;
import java.util.Map;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpUriRequest;

import com.google.common.collect.Maps;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.sap.cloud.sdk.cloudplatform.connectivity.HttpDestinationProperties;
import com.sap.cloud.sdk.datamodel.odata.helper.SingleValuedFluentHelperFunction;

/**
 * Fluent helper for the <b>Post</b> OData function import.
 *
 */
public class PostFluentHelper
    extends
    SingleValuedFluentHelperFunction<PostFluentHelper, PostInvoiceExportParameters, PostInvoiceExportParameters>
{

    private final Map<String, Object> values = Maps.newHashMap();

    /**
     * Creates a fluent helper object that will execute the <b>Post</b> OData function import with the provided
     * parameters. To perform execution, call the {@link #executeRequest executeRequest} method on the fluent helper
     * object.
     *
     * @param servicePath
     *            Service path to be used to call the functions against.
     * @param supplierInvoice
     *            Constraints: Not nullable, Maximum length: 10
     *            <p>
     *            Original parameter name from the Odata EDM: <b>SupplierInvoice</b>
     *            </p>
     * @param fiscalYear
     *            Constraints: Not nullable, Maximum length: 4
     *            <p>
     *            Original parameter name from the Odata EDM: <b>FiscalYear</b>
     *            </p>
     */
    public PostFluentHelper(
        @Nonnull final String servicePath,
        @Nonnull final String supplierInvoice,
        @Nonnull final String fiscalYear )
    {
        super(servicePath);
        values.put("SupplierInvoice", supplierInvoice);
        values.put("FiscalYear", fiscalYear);
    }

    @Override
    @Nonnull
    protected Class<PostInvoiceExportParameters> getEntityClass()
    {
        return PostInvoiceExportParameters.class;
    }

    @Override
    @Nonnull
    protected String getFunctionName()
    {
        return "Post";
    }

    @Override
    @Nullable
    protected JsonElement refineJsonResponse( @Nullable JsonElement jsonElement )
    {
        if( (jsonElement instanceof JsonObject) && ((JsonObject) jsonElement).has(getFunctionName()) ) {
            jsonElement = ((JsonObject) jsonElement).get(getFunctionName());
        }
        return super.refineJsonResponse(jsonElement);
    }

    @Override
    @Nonnull
    protected Map<String, Object> getParameters()
    {
        return values;
    }

    @Override
    @Nonnull
    protected HttpUriRequest createRequest( @Nonnull final URI uri )
    {
        return new HttpPost(uri);
    }

    /**
     * Execute this function import.
     *
     */
    @Override
    @Nullable
    public PostInvoiceExportParameters executeRequest( @Nonnull final HttpDestinationProperties destination )
    {
        return super.executeSingle(destination);
    }

}
