/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.supplierinvoiceintegration;

import java.util.Map;

import javax.annotation.Nonnull;

import com.google.common.collect.Maps;
import com.sap.cloud.sdk.datamodel.odata.helper.FluentHelperByKey;
import com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.supplierinvoiceintegration.selectable.SupplierInvoiceTaxSelectable;

/**
 * Fluent helper to fetch a single
 * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.supplierinvoiceintegration.SupplierInvoiceTax
 * SupplierInvoiceTax} entity using key fields. This fluent helper allows methods which modify the underlying query to
 * be called before executing the query itself.
 *
 */
public class SupplierInvoiceTaxByKeyFluentHelper
    extends
    FluentHelperByKey<SupplierInvoiceTaxByKeyFluentHelper, SupplierInvoiceTax, SupplierInvoiceTaxSelectable>
{

    private final Map<String, Object> key = Maps.newHashMap();

    /**
     * Creates a fluent helper object that will fetch a single
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.supplierinvoiceintegration.SupplierInvoiceTax
     * SupplierInvoiceTax} entity with the provided key field values. To perform execution, call the
     * {@link #executeRequest executeRequest} method on the fluent helper object.
     *
     * @param entityCollection
     *            Entity Collection to be used to fetch a single {@code SupplierInvoiceTax}
     * @param servicePath
     *            Service path to be used to fetch a single {@code SupplierInvoiceTax}
     * @param supplierInvoice
     *            Document Number of an Invoice Document
     *            <p>
     *            Constraints: Not nullable, Maximum length: 10
     *            </p>
     * @param taxCode
     *            Tax code
     *            <p>
     *            Constraints: Not nullable, Maximum length: 2
     *            </p>
     * @param fiscalYear
     *            Fiscal Year
     *            <p>
     *            Constraints: Not nullable, Maximum length: 4
     *            </p>
     * @param supplierInvoiceTaxCounter
     *            Document Item in Invoice Document
     *            <p>
     *            Constraints: Not nullable, Maximum length: 6
     *            </p>
     */
    public SupplierInvoiceTaxByKeyFluentHelper(
        @Nonnull final String servicePath,
        @Nonnull final String entityCollection,
        final String supplierInvoice,
        final String fiscalYear,
        final String taxCode,
        final String supplierInvoiceTaxCounter )
    {
        super(servicePath, entityCollection);
        this.key.put("SupplierInvoice", supplierInvoice);
        this.key.put("FiscalYear", fiscalYear);
        this.key.put("TaxCode", taxCode);
        this.key.put("SupplierInvoiceTaxCounter", supplierInvoiceTaxCounter);
    }

    @Override
    @Nonnull
    protected Class<SupplierInvoiceTax> getEntityClass()
    {
        return SupplierInvoiceTax.class;
    }

    @Override
    @Nonnull
    protected Map<String, Object> getKey()
    {
        return key;
    }

}
