/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.trialbalance;

import java.util.Map;

import javax.annotation.Nonnull;

import com.google.common.collect.Maps;
import com.sap.cloud.sdk.datamodel.odata.helper.FluentHelperByKey;
import com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.trialbalance.selectable.FixedAssetSelectable;

/**
 * Fluent helper to fetch a single {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.trialbalance.FixedAsset
 * FixedAsset} entity using key fields. This fluent helper allows methods which modify the underlying query to be called
 * before executing the query itself.
 *
 */
public class FixedAssetByKeyFluentHelper
    extends
    FluentHelperByKey<FixedAssetByKeyFluentHelper, FixedAsset, FixedAssetSelectable>
{

    private final Map<String, Object> key = Maps.newHashMap();

    /**
     * Creates a fluent helper object that will fetch a single
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.trialbalance.FixedAsset FixedAsset} entity with the
     * provided key field values. To perform execution, call the {@link #executeRequest executeRequest} method on the
     * fluent helper object.
     *
     * @param companyCode
     *            Asset Subnumber (Company Code)
     *            <p>
     *            Constraints: Not nullable, Maximum length: 4
     *            </p>
     * @param entityCollection
     *            Entity Collection to be used to fetch a single {@code FixedAsset}
     * @param servicePath
     *            Service path to be used to fetch a single {@code FixedAsset}
     * @param masterFixedAsset
     *            Asset Subnumber (Fixed Asset)
     *            <p>
     *            Constraints: Not nullable, Maximum length: 12
     *            </p>
     * @param fixedAsset_ID
     *            Asset Subnumber
     *            <p>
     *            Constraints: Not nullable, Maximum length: 4
     *            </p>
     */
    public FixedAssetByKeyFluentHelper(
        @Nonnull final String servicePath,
        @Nonnull final String entityCollection,
        final String fixedAsset_ID,
        final String companyCode,
        final String masterFixedAsset )
    {
        super(servicePath, entityCollection);
        this.key.put("FixedAsset_ID", fixedAsset_ID);
        this.key.put("CompanyCode", companyCode);
        this.key.put("MasterFixedAsset", masterFixedAsset);
    }

    @Override
    @Nonnull
    protected Class<FixedAsset> getEntityClass()
    {
        return FixedAsset.class;
    }

    @Override
    @Nonnull
    protected Map<String, Object> getKey()
    {
        return key;
    }

}
