/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.trialbalance;

import java.util.Map;

import javax.annotation.Nonnull;

import com.google.common.collect.Maps;
import com.sap.cloud.sdk.datamodel.odata.helper.FluentHelperByKey;
import com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.trialbalance.selectable.HouseBankAccountSelectable;

/**
 * Fluent helper to fetch a single
 * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.trialbalance.HouseBankAccount HouseBankAccount} entity
 * using key fields. This fluent helper allows methods which modify the underlying query to be called before executing
 * the query itself.
 *
 */
public class HouseBankAccountByKeyFluentHelper
    extends
    FluentHelperByKey<HouseBankAccountByKeyFluentHelper, HouseBankAccount, HouseBankAccountSelectable>
{

    private final Map<String, Object> key = Maps.newHashMap();

    /**
     * Creates a fluent helper object that will fetch a single
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.trialbalance.HouseBankAccount HouseBankAccount} entity
     * with the provided key field values. To perform execution, call the {@link #executeRequest executeRequest} method
     * on the fluent helper object.
     *
     * @param companyCode
     *            House Bank Account (Company Code)
     *            <p>
     *            Constraints: Not nullable, Maximum length: 4
     *            </p>
     * @param entityCollection
     *            Entity Collection to be used to fetch a single {@code HouseBankAccount}
     * @param servicePath
     *            Service path to be used to fetch a single {@code HouseBankAccount}
     * @param houseBankAccount_ID
     *            House Bank Account
     *            <p>
     *            Constraints: Not nullable, Maximum length: 5
     *            </p>
     * @param houseBank
     *            House Bank Account (House Bank)
     *            <p>
     *            Constraints: Not nullable, Maximum length: 5
     *            </p>
     */
    public HouseBankAccountByKeyFluentHelper(
        @Nonnull final String servicePath,
        @Nonnull final String entityCollection,
        final String houseBankAccount_ID,
        final String companyCode,
        final String houseBank )
    {
        super(servicePath, entityCollection);
        this.key.put("HouseBankAccount_ID", houseBankAccount_ID);
        this.key.put("CompanyCode", companyCode);
        this.key.put("HouseBank", houseBank);
    }

    @Override
    @Nonnull
    protected Class<HouseBankAccount> getEntityClass()
    {
        return HouseBankAccount.class;
    }

    @Override
    @Nonnull
    protected Map<String, Object> getKey()
    {
        return key;
    }

}
