/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.trialbalance;

import java.util.Map;

import javax.annotation.Nonnull;

import com.google.common.collect.Maps;
import com.sap.cloud.sdk.datamodel.odata.helper.FluentHelperByKey;
import com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.trialbalance.selectable.HouseBankSelectable;

/**
 * Fluent helper to fetch a single {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.trialbalance.HouseBank
 * HouseBank} entity using key fields. This fluent helper allows methods which modify the underlying query to be called
 * before executing the query itself.
 *
 */
public class HouseBankByKeyFluentHelper
    extends
    FluentHelperByKey<HouseBankByKeyFluentHelper, HouseBank, HouseBankSelectable>
{

    private final Map<String, Object> key = Maps.newHashMap();

    /**
     * Creates a fluent helper object that will fetch a single
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.trialbalance.HouseBank HouseBank} entity with the
     * provided key field values. To perform execution, call the {@link #executeRequest executeRequest} method on the
     * fluent helper object.
     *
     * @param companyCode
     *            House Bank (Company Code)
     *            <p>
     *            Constraints: Not nullable, Maximum length: 4
     *            </p>
     * @param entityCollection
     *            Entity Collection to be used to fetch a single {@code HouseBank}
     * @param houseBank_ID
     *            House Bank
     *            <p>
     *            Constraints: Not nullable, Maximum length: 5
     *            </p>
     * @param servicePath
     *            Service path to be used to fetch a single {@code HouseBank}
     */
    public HouseBankByKeyFluentHelper(
        @Nonnull final String servicePath,
        @Nonnull final String entityCollection,
        final String houseBank_ID,
        final String companyCode )
    {
        super(servicePath, entityCollection);
        this.key.put("HouseBank_ID", houseBank_ID);
        this.key.put("CompanyCode", companyCode);
    }

    @Override
    @Nonnull
    protected Class<HouseBank> getEntityClass()
    {
        return HouseBank.class;
    }

    @Override
    @Nonnull
    protected Map<String, Object> getKey()
    {
        return key;
    }

}
