/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.trialbalance;

import java.util.Map;

import javax.annotation.Nonnull;

import com.google.common.collect.Maps;
import com.sap.cloud.sdk.datamodel.odata.helper.FluentHelperByKey;
import com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.trialbalance.selectable.SalesDocumentItemSelectable;

/**
 * Fluent helper to fetch a single
 * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.trialbalance.SalesDocumentItem SalesDocumentItem} entity
 * using key fields. This fluent helper allows methods which modify the underlying query to be called before executing
 * the query itself.
 *
 */
public class SalesDocumentItemByKeyFluentHelper
    extends
    FluentHelperByKey<SalesDocumentItemByKeyFluentHelper, SalesDocumentItem, SalesDocumentItemSelectable>
{

    private final Map<String, Object> key = Maps.newHashMap();

    /**
     * Creates a fluent helper object that will fetch a single
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.trialbalance.SalesDocumentItem SalesDocumentItem}
     * entity with the provided key field values. To perform execution, call the {@link #executeRequest executeRequest}
     * method on the fluent helper object.
     *
     * @param entityCollection
     *            Entity Collection to be used to fetch a single {@code SalesDocumentItem}
     * @param salesDocument
     *            Sales Document Item (Sales Document)
     *            <p>
     *            Constraints: Not nullable, Maximum length: 10
     *            </p>
     * @param servicePath
     *            Service path to be used to fetch a single {@code SalesDocumentItem}
     * @param salesDocumentItem_ID
     *            Sales Document Item
     *            <p>
     *            Constraints: Not nullable, Maximum length: 6
     *            </p>
     */
    public SalesDocumentItemByKeyFluentHelper(
        @Nonnull final String servicePath,
        @Nonnull final String entityCollection,
        final String salesDocumentItem_ID,
        final String salesDocument )
    {
        super(servicePath, entityCollection);
        this.key.put("SalesDocumentItem_ID", salesDocumentItem_ID);
        this.key.put("SalesDocument", salesDocument);
    }

    @Override
    @Nonnull
    protected Class<SalesDocumentItem> getEntityClass()
    {
        return SalesDocumentItem.class;
    }

    @Override
    @Nonnull
    protected Map<String, Object> getKey()
    {
        return key;
    }

}
