/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.trialbalance;

import java.util.Map;

import javax.annotation.Nonnull;

import com.google.common.collect.Maps;
import com.sap.cloud.sdk.datamodel.odata.helper.FluentHelperByKey;
import com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.trialbalance.selectable.SalesOrderItemSelectable;

/**
 * Fluent helper to fetch a single
 * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.trialbalance.SalesOrderItem SalesOrderItem} entity using
 * key fields. This fluent helper allows methods which modify the underlying query to be called before executing the
 * query itself.
 *
 */
public class SalesOrderItemByKeyFluentHelper
    extends
    FluentHelperByKey<SalesOrderItemByKeyFluentHelper, SalesOrderItem, SalesOrderItemSelectable>
{

    private final Map<String, Object> key = Maps.newHashMap();

    /**
     * Creates a fluent helper object that will fetch a single
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.trialbalance.SalesOrderItem SalesOrderItem} entity
     * with the provided key field values. To perform execution, call the {@link #executeRequest executeRequest} method
     * on the fluent helper object.
     *
     * @param entityCollection
     *            Entity Collection to be used to fetch a single {@code SalesOrderItem}
     * @param servicePath
     *            Service path to be used to fetch a single {@code SalesOrderItem}
     * @param salesOrderItem_ID
     *            Item
     *            <p>
     *            Constraints: Not nullable, Maximum length: 6
     *            </p>
     * @param salesOrder
     *            Item (Sales Order)
     *            <p>
     *            Constraints: Not nullable, Maximum length: 10
     *            </p>
     */
    public SalesOrderItemByKeyFluentHelper(
        @Nonnull final String servicePath,
        @Nonnull final String entityCollection,
        final String salesOrderItem_ID,
        final String salesOrder )
    {
        super(servicePath, entityCollection);
        this.key.put("SalesOrderItem_ID", salesOrderItem_ID);
        this.key.put("SalesOrder", salesOrder);
    }

    @Override
    @Nonnull
    protected Class<SalesOrderItem> getEntityClass()
    {
        return SalesOrderItem.class;
    }

    @Override
    @Nonnull
    protected Map<String, Object> getKey()
    {
        return key;
    }

}
