/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.warehouse;

import java.util.Map;

import javax.annotation.Nonnull;

import com.google.common.collect.Maps;
import com.sap.cloud.sdk.datamodel.odata.helper.FluentHelperByKey;
import com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.warehouse.selectable.WarehouseStorageTypeSelectable;

/**
 * Fluent helper to fetch a single
 * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.warehouse.WarehouseStorageType WarehouseStorageType}
 * entity using key fields. This fluent helper allows methods which modify the underlying query to be called before
 * executing the query itself.
 *
 */
public class WarehouseStorageTypeByKeyFluentHelper
    extends
    FluentHelperByKey<WarehouseStorageTypeByKeyFluentHelper, WarehouseStorageType, WarehouseStorageTypeSelectable>
{

    private final Map<String, Object> key = Maps.newHashMap();

    /**
     * Creates a fluent helper object that will fetch a single
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.warehouse.WarehouseStorageType WarehouseStorageType}
     * entity with the provided key field values. To perform execution, call the {@link #executeRequest executeRequest}
     * method on the fluent helper object.
     *
     * @param entityCollection
     *            Entity Collection to be used to fetch a single {@code WarehouseStorageType}
     * @param servicePath
     *            Service path to be used to fetch a single {@code WarehouseStorageType}
     * @param storageType
     *            Storage Type
     *            <p>
     *            Constraints: Not nullable, Maximum length: 4
     *            </p>
     * @param warehouse
     *            Warehouse Number/Warehouse Complex
     *            <p>
     *            Constraints: Not nullable, Maximum length: 4
     *            </p>
     */
    public WarehouseStorageTypeByKeyFluentHelper(
        @Nonnull final String servicePath,
        @Nonnull final String entityCollection,
        final String warehouse,
        final String storageType )
    {
        super(servicePath, entityCollection);
        this.key.put("Warehouse", warehouse);
        this.key.put("StorageType", storageType);
    }

    @Override
    @Nonnull
    protected Class<WarehouseStorageType> getEntityClass()
    {
        return WarehouseStorageType.class;
    }

    @Override
    @Nonnull
    protected Map<String, Object> getKey()
    {
        return key;
    }

}
