/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.warehouse;

import java.util.Map;

import javax.annotation.Nonnull;

import com.google.common.collect.Maps;
import com.sap.cloud.sdk.datamodel.odata.helper.FluentHelperByKey;
import com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.warehouse.selectable.WarehouseStorageTypeTextSelectable;

/**
 * Fluent helper to fetch a single
 * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.warehouse.WarehouseStorageTypeText
 * WarehouseStorageTypeText} entity using key fields. This fluent helper allows methods which modify the underlying
 * query to be called before executing the query itself.
 *
 */
public class WarehouseStorageTypeTextByKeyFluentHelper
    extends
    FluentHelperByKey<WarehouseStorageTypeTextByKeyFluentHelper, WarehouseStorageTypeText, WarehouseStorageTypeTextSelectable>
{

    private final Map<String, Object> key = Maps.newHashMap();

    /**
     * Creates a fluent helper object that will fetch a single
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.warehouse.WarehouseStorageTypeText
     * WarehouseStorageTypeText} entity with the provided key field values. To perform execution, call the
     * {@link #executeRequest executeRequest} method on the fluent helper object.
     *
     * @param entityCollection
     *            Entity Collection to be used to fetch a single {@code WarehouseStorageTypeText}
     * @param servicePath
     *            Service path to be used to fetch a single {@code WarehouseStorageTypeText}
     * @param storageType
     *            Storage Type
     *            <p>
     *            Constraints: Not nullable, Maximum length: 4
     *            </p>
     * @param language
     *            Language Key
     *            <p>
     *            Constraints: Not nullable, Maximum length: 2
     *            </p>
     * @param warehouse
     *            Warehouse Number/Warehouse Complex
     *            <p>
     *            Constraints: Not nullable, Maximum length: 4
     *            </p>
     */
    public WarehouseStorageTypeTextByKeyFluentHelper(
        @Nonnull final String servicePath,
        @Nonnull final String entityCollection,
        final String language,
        final String warehouse,
        final String storageType )
    {
        super(servicePath, entityCollection);
        this.key.put("Language", language);
        this.key.put("Warehouse", warehouse);
        this.key.put("StorageType", storageType);
    }

    @Override
    @Nonnull
    protected Class<WarehouseStorageTypeText> getEntityClass()
    {
        return WarehouseStorageTypeText.class;
    }

    @Override
    @Nonnull
    protected Map<String, Object> getKey()
    {
        return key;
    }

}
