/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.warehouseresource;

import java.net.URI;
import java.util.Map;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpUriRequest;

import com.google.common.collect.Maps;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.sap.cloud.sdk.cloudplatform.connectivity.HttpDestinationProperties;
import com.sap.cloud.sdk.datamodel.odata.helper.SingleValuedFluentHelperFunction;

/**
 * Fluent helper for the <b>LogoffFromWarehouseResource</b> OData function import.
 *
 */
public class LogoffFromWarehouseResourceFluentHelper
    extends
    SingleValuedFluentHelperFunction<LogoffFromWarehouseResourceFluentHelper, WarehouseResource, WarehouseResource>
{

    private final Map<String, Object> values = Maps.newHashMap();

    /**
     * Creates a fluent helper object that will execute the <b>LogoffFromWarehouseResource</b> OData function import
     * with the provided parameters. To perform execution, call the {@link #executeRequest executeRequest} method on the
     * fluent helper object.
     *
     * @param servicePath
     *            Service path to be used to call the functions against.
     * @param warehouseResource
     *            Resource
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Not nullable, Maximum length: 18
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>WarehouseResource</b>
     *            </p>
     * @param warehouse
     *            Warehouse Number
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Not nullable, Maximum length: 4
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>Warehouse</b>
     *            </p>
     */
    public LogoffFromWarehouseResourceFluentHelper(
        @Nonnull final String servicePath,
        @Nonnull final String warehouse,
        @Nonnull final String warehouseResource )
    {
        super(servicePath);
        values.put("Warehouse", warehouse);
        values.put("WarehouseResource", warehouseResource);
    }

    @Override
    @Nonnull
    protected Class<WarehouseResource> getEntityClass()
    {
        return WarehouseResource.class;
    }

    @Override
    @Nonnull
    protected String getFunctionName()
    {
        return "LogoffFromWarehouseResource";
    }

    @Override
    @Nullable
    protected JsonElement refineJsonResponse( @Nullable JsonElement jsonElement )
    {
        if( (jsonElement instanceof JsonObject) && ((JsonObject) jsonElement).has(getFunctionName()) ) {
            jsonElement = ((JsonObject) jsonElement).get(getFunctionName());
        }
        return super.refineJsonResponse(jsonElement);
    }

    @Override
    @Nonnull
    protected Map<String, Object> getParameters()
    {
        return values;
    }

    @Override
    @Nonnull
    protected HttpUriRequest createRequest( @Nonnull final URI uri )
    {
        return new HttpPost(uri);
    }

    /**
     * Execute this function import.
     *
     */
    @Override
    @Nullable
    public WarehouseResource executeRequest( @Nonnull final HttpDestinationProperties destination )
    {
        return super.executeSingle(destination);
    }

}
