/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.warehousetask;

import java.net.URI;
import java.util.Map;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpUriRequest;

import com.google.common.collect.Maps;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.sap.cloud.sdk.cloudplatform.connectivity.HttpDestinationProperties;
import com.sap.cloud.sdk.datamodel.odata.helper.SingleValuedFluentHelperFunction;

/**
 * Fluent helper for the <b>ConfirmWarehouseTaskExact</b> OData function import.
 *
 */
public class ConfirmWarehouseTaskExactFluentHelper
    extends
    SingleValuedFluentHelperFunction<ConfirmWarehouseTaskExactFluentHelper, DummyFunctionImportResult, DummyFunctionImportResult>
{

    private final Map<String, Object> values = Maps.newHashMap();

    /**
     * Creates a fluent helper object that will execute the <b>ConfirmWarehouseTaskExact</b> OData function import with
     * the provided parameters. To perform execution, call the {@link #executeRequest executeRequest} method on the
     * fluent helper object.
     *
     * @param warehouseTaskItem
     *            WT Item
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Not nullable, Maximum length: 4
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>WarehouseTaskItem</b>
     *            </p>
     * @param servicePath
     *            Service path to be used to call the functions against.
     * @param warehouse
     *            Warehouse Number
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Not nullable, Maximum length: 4
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>Warehouse</b>
     *            </p>
     * @param warehouseTask
     *            Warehouse Task
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Not nullable, Maximum length: 12
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>WarehouseTask</b>
     *            </p>
     */
    public ConfirmWarehouseTaskExactFluentHelper(
        @Nonnull final String servicePath,
        @Nonnull final String warehouse,
        @Nonnull final String warehouseTask,
        @Nonnull final String warehouseTaskItem )
    {
        super(servicePath);
        values.put("Warehouse", warehouse);
        values.put("WarehouseTask", warehouseTask);
        values.put("WarehouseTaskItem", warehouseTaskItem);
    }

    @Override
    @Nonnull
    protected Class<DummyFunctionImportResult> getEntityClass()
    {
        return DummyFunctionImportResult.class;
    }

    @Override
    @Nonnull
    protected String getFunctionName()
    {
        return "ConfirmWarehouseTaskExact";
    }

    @Override
    @Nullable
    protected JsonElement refineJsonResponse( @Nullable JsonElement jsonElement )
    {
        if( (jsonElement instanceof JsonObject) && ((JsonObject) jsonElement).has(getFunctionName()) ) {
            jsonElement = ((JsonObject) jsonElement).get(getFunctionName());
        }
        return super.refineJsonResponse(jsonElement);
    }

    @Override
    @Nonnull
    protected Map<String, Object> getParameters()
    {
        return values;
    }

    @Override
    @Nonnull
    protected HttpUriRequest createRequest( @Nonnull final URI uri )
    {
        return new HttpPost(uri);
    }

    /**
     * Execute this function import.
     *
     */
    @Override
    @Nullable
    public DummyFunctionImportResult executeRequest( @Nonnull final HttpDestinationProperties destination )
    {
        return super.executeSingle(destination);
    }

}
