/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.warehousetask;

import java.net.URI;
import java.util.Map;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpUriRequest;

import com.google.common.collect.Maps;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.sap.cloud.sdk.cloudplatform.connectivity.HttpDestinationProperties;
import com.sap.cloud.sdk.datamodel.odata.helper.SingleValuedFluentHelperFunction;

/**
 * Fluent helper for the <b>ConfirmWarehouseTaskHU</b> OData function import.
 *
 */
public class ConfirmWarehouseTaskHUFluentHelper
    extends
    SingleValuedFluentHelperFunction<ConfirmWarehouseTaskHUFluentHelper, DummyFunctionImportResult, DummyFunctionImportResult>
{

    private final Map<String, Object> values = Maps.newHashMap();

    /**
     * Creates a fluent helper object that will execute the <b>ConfirmWarehouseTaskHU</b> OData function import with the
     * provided parameters. To perform execution, call the {@link #executeRequest executeRequest} method on the fluent
     * helper object.
     *
     * @param warehouseTaskItem
     *            WT Item
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Not nullable, Maximum length: 4
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>WarehouseTaskItem</b>
     *            </p>
     * @param destinationStorageBin
     *            Destination Bin
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Nullable, Maximum length: 18
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>DestinationStorageBin</b>
     *            </p>
     * @param servicePath
     *            Service path to be used to call the functions against.
     * @param warehouse
     *            Warehouse Number
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Not nullable, Maximum length: 4
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>Warehouse</b>
     *            </p>
     * @param warehouseTask
     *            Warehouse Task
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Not nullable, Maximum length: 12
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>WarehouseTask</b>
     *            </p>
     * @param whseTaskExCodeDestStorageBin
     *            Exception Code
     *            <p>
     *            </p>
     *            <p>
     *            Constraints: Nullable, Maximum length: 4
     *            </p>
     *            <p>
     *            Original parameter name from the Odata EDM: <b>WhseTaskExCodeDestStorageBin</b>
     *            </p>
     */
    public ConfirmWarehouseTaskHUFluentHelper(
        @Nonnull final String servicePath,
        @Nonnull final String warehouse,
        @Nonnull final String warehouseTask,
        @Nonnull final String warehouseTaskItem,
        @Nullable final String destinationStorageBin,
        @Nullable final String whseTaskExCodeDestStorageBin )
    {
        super(servicePath);
        values.put("Warehouse", warehouse);
        values.put("WarehouseTask", warehouseTask);
        values.put("WarehouseTaskItem", warehouseTaskItem);
        if( destinationStorageBin != null ) {
            values.put("DestinationStorageBin", destinationStorageBin);
        }
        if( whseTaskExCodeDestStorageBin != null ) {
            values.put("WhseTaskExCodeDestStorageBin", whseTaskExCodeDestStorageBin);
        }
    }

    @Override
    @Nonnull
    protected Class<DummyFunctionImportResult> getEntityClass()
    {
        return DummyFunctionImportResult.class;
    }

    @Override
    @Nonnull
    protected String getFunctionName()
    {
        return "ConfirmWarehouseTaskHU";
    }

    @Override
    @Nullable
    protected JsonElement refineJsonResponse( @Nullable JsonElement jsonElement )
    {
        if( (jsonElement instanceof JsonObject) && ((JsonObject) jsonElement).has(getFunctionName()) ) {
            jsonElement = ((JsonObject) jsonElement).get(getFunctionName());
        }
        return super.refineJsonResponse(jsonElement);
    }

    @Override
    @Nonnull
    protected Map<String, Object> getParameters()
    {
        return values;
    }

    @Override
    @Nonnull
    protected HttpUriRequest createRequest( @Nonnull final URI uri )
    {
        return new HttpPost(uri);
    }

    /**
     * Execute this function import.
     *
     */
    @Override
    @Nullable
    public DummyFunctionImportResult executeRequest( @Nonnull final HttpDestinationProperties destination )
    {
        return super.executeSingle(destination);
    }

}
