/*
 * Copyright (c) 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.warehousetask;

import java.util.Map;

import javax.annotation.Nonnull;

import com.google.common.collect.Maps;
import com.sap.cloud.sdk.datamodel.odata.helper.FluentHelperByKey;
import com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.warehousetask.selectable.WarehouseTaskExceptionCodeSelectable;

/**
 * Fluent helper to fetch a single
 * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.warehousetask.WarehouseTaskExceptionCode
 * WarehouseTaskExceptionCode} entity using key fields. This fluent helper allows methods which modify the underlying
 * query to be called before executing the query itself.
 *
 */
public class WarehouseTaskExceptionCodeByKeyFluentHelper
    extends
    FluentHelperByKey<WarehouseTaskExceptionCodeByKeyFluentHelper, WarehouseTaskExceptionCode, WarehouseTaskExceptionCodeSelectable>
{

    private final Map<String, Object> key = Maps.newHashMap();

    /**
     * Creates a fluent helper object that will fetch a single
     * {@link com.sap.cloud.sdk.s4hana.datamodel.odata.namespaces.warehousetask.WarehouseTaskExceptionCode
     * WarehouseTaskExceptionCode} entity with the provided key field values. To perform execution, call the
     * {@link #executeRequest executeRequest} method on the fluent helper object.
     *
     * @param entityCollection
     *            Entity Collection to be used to fetch a single {@code WarehouseTaskExceptionCode}
     * @param businessContextExecutionStep
     *            Execution Step in Business Context
     *            <p>
     *            Constraints: Not nullable, Maximum length: 2
     *            </p>
     * @param warehouseTaskItem
     *            Warehouse Task Item
     *            <p>
     *            Constraints: Not nullable, Maximum length: 4
     *            </p>
     * @param warehouseTaskExceptionCode
     *            Exception Code
     *            <p>
     *            Constraints: Not nullable, Maximum length: 4
     *            </p>
     * @param servicePath
     *            Service path to be used to fetch a single {@code WarehouseTaskExceptionCode}
     * @param businessContext
     *            Business Context
     *            <p>
     *            Constraints: Not nullable, Maximum length: 3
     *            </p>
     * @param warehouse
     *            Warehouse Number/Warehouse Complex
     *            <p>
     *            Constraints: Not nullable, Maximum length: 4
     *            </p>
     * @param warehouseTask
     *            Warehouse Task
     *            <p>
     *            Constraints: Not nullable, Maximum length: 12
     *            </p>
     */
    public WarehouseTaskExceptionCodeByKeyFluentHelper(
        @Nonnull final String servicePath,
        @Nonnull final String entityCollection,
        final String warehouse,
        final String warehouseTask,
        final String warehouseTaskItem,
        final String warehouseTaskExceptionCode,
        final String businessContext,
        final String businessContextExecutionStep )
    {
        super(servicePath, entityCollection);
        this.key.put("Warehouse", warehouse);
        this.key.put("WarehouseTask", warehouseTask);
        this.key.put("WarehouseTaskItem", warehouseTaskItem);
        this.key.put("WarehouseTaskExceptionCode", warehouseTaskExceptionCode);
        this.key.put("BusinessContext", businessContext);
        this.key.put("BusinessContextExecutionStep", businessContextExecutionStep);
    }

    @Override
    @Nonnull
    protected Class<WarehouseTaskExceptionCode> getEntityClass()
    {
        return WarehouseTaskExceptionCode.class;
    }

    @Override
    @Nonnull
    protected Map<String, Object> getKey()
    {
        return key;
    }

}
