/*
 * Copyright (c) 2021 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.connectivity.exception;

import java.util.Set;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import com.google.common.collect.Sets;
import com.sap.cloud.sdk.cloudplatform.security.Authorization;
import com.sap.cloud.sdk.cloudplatform.security.principal.Principal;
import com.sap.cloud.sdk.cloudplatform.servlet.response.AccessDeniedResponse;
import com.sap.cloud.sdk.cloudplatform.servlet.response.ResponseWithErrorCode;

import lombok.Getter;
import lombok.NoArgsConstructor;

/**
 * Thrown when a certain service denies access to the requested resources.
 */
@NoArgsConstructor
public class AccessDeniedException extends RequestExecutionException
{
    private static final long serialVersionUID = 8256471661877663966L;

    /**
     * Static factory method to instantiate a new exception.
     * 
     * @param principal
     *            The principal reference.
     * @param missingAuthorizations
     *            The missing authorization values.
     * @return The newly created exception.
     */
    @Nonnull
    public static AccessDeniedException raiseMissingAuthorizations(
        @Nullable final Principal principal,
        @Nullable final Iterable<? extends Authorization> missingAuthorizations )
    {
        return new AccessDeniedException(
            principal,
            missingAuthorizations != null ? Sets.newHashSet(missingAuthorizations) : null);
    }

    /**
     * The principal reference.
     */
    @Getter
    @Nullable
    protected transient Principal principal = null;

    /**
     * The missing authorization values.
     */
    @Getter
    @Nullable
    protected transient Set<Authorization> missingAuthorizations = null;

    /**
     * Constructor.
     * 
     * @param message
     *            The message.
     */
    public AccessDeniedException( @Nullable final String message )
    {
        super(message);
    }

    /**
     * Constructor.
     * 
     * @param cause
     *            The error cause.
     */
    public AccessDeniedException( @Nullable final Throwable cause )
    {
        super(cause);
    }

    /**
     * Constructor.
     * 
     * @param message
     *            The message.
     * @param cause
     *            The error cause.
     */
    public AccessDeniedException( @Nullable final String message, @Nullable final Throwable cause )
    {
        super(message, cause);
    }

    /**
     * Constructor.
     * 
     * @param principal
     *            The principal reference.
     * @param missingAuthorizations
     *            The missing authorization values.
     */
    public AccessDeniedException(
        @Nullable final Principal principal,
        @Nullable final Set<Authorization> missingAuthorizations )
    {
        super(buildErrorMessage(principal, missingAuthorizations));

        this.principal = principal;
        this.missingAuthorizations = missingAuthorizations;
    }

    private static String buildErrorMessage(
        @Nullable final Principal principal,
        @Nullable final Iterable<? extends Authorization> missingAuthorizations )
    {
        if( principal != null ) {
            return "Principal "
                + principal
                + " does not have the required authorizations"
                + (missingAuthorizations != null ? ": " + missingAuthorizations : "")
                + ".";
        } else {
            return "Missing authorizations" + (missingAuthorizations != null ? ": " + missingAuthorizations : "") + ".";
        }
    }

    @Nonnull
    @Override
    public ResponseWithErrorCode getErrorResponse()
    {
        return new AccessDeniedResponse(null, null, "Access denied.");
    }
}
