/*
 * Copyright (c) 2022 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.connectivity;

import static com.sap.cloud.sdk.s4hana.connectivity.ErpHttpDestinationUtils.asErp;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.URI;
import java.nio.charset.StandardCharsets;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.zip.GZIPOutputStream;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.apache.http.HttpEntityEnclosingRequest;
import org.apache.http.HttpHeaders;
import org.apache.http.HttpResponse;
import org.apache.http.HttpStatus;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpDelete;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpHead;
import org.apache.http.client.methods.HttpOptions;
import org.apache.http.client.methods.HttpPatch;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpPut;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.entity.ByteArrayEntity;

import com.google.common.base.Strings;
import com.google.common.collect.Lists;
import com.sap.cloud.sdk.cloudplatform.auditlog.AccessedAttribute;
import com.sap.cloud.sdk.cloudplatform.auditlog.AuditLogger;
import com.sap.cloud.sdk.cloudplatform.auditlog.AuditedDataObject;
import com.sap.cloud.sdk.cloudplatform.auditlog.AuditedDataSubject;
import com.sap.cloud.sdk.cloudplatform.connectivity.Header;
import com.sap.cloud.sdk.cloudplatform.connectivity.HttpClientAccessor;
import com.sap.cloud.sdk.cloudplatform.connectivity.HttpDestination;
import com.sap.cloud.sdk.cloudplatform.connectivity.HttpEntityUtil;
import com.sap.cloud.sdk.cloudplatform.connectivity.exception.DestinationAccessException;
import com.sap.cloud.sdk.cloudplatform.connectivity.exception.DestinationNotFoundException;
import com.sap.cloud.sdk.cloudplatform.exception.ShouldNotHappenException;
import com.sap.cloud.sdk.cloudplatform.security.Authorization;
import com.sap.cloud.sdk.s4hana.connectivity.exception.AccessDeniedException;
import com.sap.cloud.sdk.s4hana.connectivity.exception.CloudConnectorException;
import com.sap.cloud.sdk.s4hana.connectivity.exception.LogonErrorException;
import com.sap.cloud.sdk.s4hana.connectivity.exception.RequestExecutionException;
import com.sap.cloud.sdk.s4hana.connectivity.exception.RequestSerializationException;
import com.sap.cloud.sdk.s4hana.serialization.SapClient;

import lombok.Getter;
import lombok.Value;
import lombok.extern.slf4j.Slf4j;

/**
 * A collection of methods which are commonly called during executions of a request against an ERP system.
 *
 * @param <RequestT>
 *            The type of the request to execute.
 * @param <RequestResultT>
 *            The type of the result to return.
 */
@Slf4j
public class ErpHttpRequestExecutor<RequestT extends Request<RequestT, RequestResultT>, RequestResultT extends RequestResult<RequestT, RequestResultT>>
{
    private static final int MAX_UNCOMPRESSED_PAYLOAD_LENGTH = 1400;

    @Getter
    private final RequestExecutionMeasurements measurements = new RequestExecutionMeasurements();

    @Nonnull
    private ByteArrayEntity getBodyAsCompressedEntity( @Nonnull final String body )
        throws RequestSerializationException
    {
        final ByteArrayEntity entity;

        final byte[] content;
        try {
            content = body.getBytes(StandardCharsets.UTF_8.toString());
        }
        catch( final UnsupportedEncodingException e ) {
            throw new RequestSerializationException("Failed to to convert payload from String to UTF8 byte[].", e);
        }

        if( content.length > MAX_UNCOMPRESSED_PAYLOAD_LENGTH ) {
            final ByteArrayOutputStream outputStream = new ByteArrayOutputStream();

            try( GZIPOutputStream gzipOutputStream = new GZIPOutputStream(outputStream) ) {
                gzipOutputStream.write(content);
            }
            catch( final IOException e ) {
                throw new RequestSerializationException("Failed to write to GZIP-compressed stream.", e);
            }

            entity = new ByteArrayEntity(outputStream.toByteArray());
            entity.setContentEncoding("gzip");

            if( log.isInfoEnabled() ) {
                log.info(
                    "Compressed length of ERP request body: "
                        + entity.getContentLength()
                        + " bytes, was "
                        + content.length
                        + " bytes.");
            }
        } else {
            entity = new ByteArrayEntity(content);
            entity.setContentEncoding(StandardCharsets.UTF_8.toString());

            if( log.isInfoEnabled() ) {
                log.info("Length of ERP request body: " + entity.getContentLength() + " bytes.");
            }
        }

        return entity;
    }

    private void handleHttpStatus(
        @Nonnull final ErpHttpDestination destination,
        final int statusCode,
        @Nullable final String responseBody,
        @Nonnull final List<Header> responseHeaders )
        throws RequestExecutionException
    {
        if( statusCode == HttpStatus.SC_OK ) {
            if( log.isTraceEnabled() ) {
                log.trace(
                    "Request execution finished successfully. Response body: "
                        + responseBody
                        + " Headers: "
                        + getNonSensitiveHeadersAsString(responseHeaders)
                        + ".");
            }
        } else {
            handleHttpError(destination, statusCode, responseBody, responseHeaders);
        }
    }

    private void handleHttpError(
        @Nonnull final ErpHttpDestination destination,
        final int statusCode,
        @Nullable final String responseBody,
        @Nonnull final List<Header> responseHeaders )
        throws RequestExecutionException
    {
        switch( statusCode ) {
            case HttpStatus.SC_UNAUTHORIZED:
                handleUnauthorized(responseBody, responseHeaders);
                return;

            case HttpStatus.SC_FORBIDDEN:
                handleForbidden(responseBody, responseHeaders);
                return;

            case HttpStatus.SC_INTERNAL_SERVER_ERROR:
                handleInternalServerError(responseBody, responseHeaders);
                return;

            case HttpStatus.SC_SERVICE_UNAVAILABLE:
                handleServiceUnavailableError(destination, responseBody, responseHeaders);
                return;

            case HttpStatus.SC_BAD_GATEWAY:
                handleBadGateway(responseBody, responseHeaders);
                return;

            default: {
                final String message =
                    "Request execution failed with status code "
                        + statusCode
                        + ". Response body: "
                        + responseBody
                        + " Headers: "
                        + getNonSensitiveHeadersAsString(responseHeaders)
                        + ".";

                throw new RequestExecutionException(message);
            }
        }
    }

    private void handleUnauthorized( @Nullable final String responseBody, @Nonnull final List<Header> responseHeaders )
        throws LogonErrorException
    {
        final String message =
            HttpStatus.SC_UNAUTHORIZED
                + " Unauthorized. The connection attempt was refused. Response body: "
                + responseBody
                + " Headers: "
                + getNonSensitiveHeadersAsString(responseHeaders)
                + ".";

        throw new LogonErrorException(message);
    }

    @Nullable
    private String getMissingAuthorization( @Nonnull final List<Header> responseHeaders )
    {
        for( final Header header : responseHeaders ) {
            if( header.getName().equals("failed-authorization-object") ) {
                return header.getValue();
            }
        }
        return null;
    }

    private void handleForbidden( @Nullable final String responseBody, @Nonnull final List<Header> responseHeaders )
        throws AccessDeniedException
    {
        final String prefix = HttpStatus.SC_FORBIDDEN + " Forbidden. ";

        if( responseBody != null && responseBody.startsWith("CX_FINS_MAP_NO_AUTH_QUERY_EXEC") ) {
            @Nullable
            final String missingAuthorization = getMissingAuthorization(responseHeaders);

            throw AccessDeniedException.raiseMissingAuthorizations(
                null,
                missingAuthorization != null
                    ? Collections.singleton(new ErpAuthorization(missingAuthorization))
                    : null);
        }

        final String message =
            prefix
                + "Failed to establish a trusted connection to the ERP. This may be caused by "
                + "a misconfiguration of the SAP Cloud Connector or a misconfiguration "
                + "of the trust certificate. Response body: "
                + responseBody
                + " Headers: "
                + getNonSensitiveHeadersAsString(responseHeaders)
                + ".";

        throw new AccessDeniedException(message);
    }

    private
        void
        handleInternalServerError( @Nullable final String responseBody, @Nonnull final List<Header> responseHeaders )
            throws RequestExecutionException
    {
        final String prefix = HttpStatus.SC_INTERNAL_SERVER_ERROR + " Internal Server Error. ";

        if( responseBody != null && responseBody.contains("ICF") && responseBody.contains("HCPMAPBM") ) {
            final String message =
                prefix
                    + "Failed to invoke ICF service. Does the user have authorization HCPMAPBM? "
                    + "Response body: "
                    + responseBody
                    + " Headers: "
                    + getNonSensitiveHeadersAsString(responseHeaders)
                    + ".";

            throw new AccessDeniedException(message);
        }

        final String message =
            prefix
                + "Request execution failed with unexpected error. Response body: "
                + responseBody
                + " Headers: "
                + getNonSensitiveHeadersAsString(responseHeaders)
                + ".";

        throw new RequestExecutionException(message);
    }

    private void handleServiceUnavailableError(
        @Nonnull final ErpHttpDestination destination,
        @Nullable final String responseBody,
        @Nonnull final List<Header> responseHeaders )
        throws RequestExecutionException
    {
        if( responseBody != null && responseBody.contains("No tunnels subscribed for tunnelId") ) {
            final String message =
                HttpStatus.SC_SERVICE_UNAVAILABLE
                    + " Service Unavailable. Failed to connect to ERP system. "
                    + "Please check the configuration of destination '"
                    + destination.get("Name").getOrElse("")
                    + "'. In an on-premise setup, ensure that the cloud connector is connected.";

            throw new CloudConnectorException(HttpStatus.SC_SERVICE_UNAVAILABLE, message);
        } else {
            handleInternalServerError(responseBody, responseHeaders);
        }
    }

    private void handleBadGateway( @Nullable final String responseBody, @Nonnull final List<Header> responseHeaders )
        throws RequestExecutionException
    {
        if( responseBody != null && responseBody.contains("Unable to open connection to backend system") ) {
            final String message =
                HttpStatus.SC_BAD_GATEWAY
                    + " Bad Gateway. Cloud connector failed to open connection to backend system. "
                    + "Is the internal host configured correc   tly? Response body: "
                    + responseBody
                    + " Headers: "
                    + getNonSensitiveHeadersAsString(responseHeaders)
                    + ".";

            throw new CloudConnectorException(HttpStatus.SC_BAD_GATEWAY, message);
        } else {
            handleInternalServerError(responseBody, responseHeaders);
        }
    }

    /**
     * Converts the given headers to a String while omitting sensitive headers to avoid leaking them to logs.
     */
    @Nonnull
    private String getNonSensitiveHeadersAsString( @Nonnull final List<Header> headers )
    {
        final StringBuilder sb = new StringBuilder();
        final Iterator<Header> headerIt = headers.iterator();

        while( headerIt.hasNext() ) {
            final Header header = headerIt.next();

            final String name = header.getName();
            String value = header.getValue();

            if( "set-cookie".equalsIgnoreCase(name) || "authorization".equalsIgnoreCase(name) ) {
                value = "(hidden)";
            }

            sb.append(name).append(": ").append(value).append(headerIt.hasNext() ? ", " : "");
        }

        return sb.toString();
    }

    /**
     * Serializes the given request, executes it, and the deserializes the response.
     *
     * @param destination
     *            The {@code HttpDestination} of this call.
     * @param request
     *            The {@code Request} to be executed.
     * @param requestSerializer
     *            The {@code RequestSerializer} to be used to write the request and read the response.
     * @return The body of the response received by the given request.
     * @throws RequestSerializationException
     *             If the request could not be serialized
     * @throws RequestExecutionException
     *             If any Exception occured during execution of the request.
     * @throws DestinationNotFoundException
     *             If the Destination cannot be found.
     * @throws DestinationAccessException
     *             If the destination is not of type DestinationType.HTTP or there is an issue while accessing
     *             destination information.
     */
    @Nonnull
    public RequestResultT execute(
        @Nonnull final HttpDestination destination,
        @Nonnull final RequestT request,
        @Nonnull final RequestSerializer<RequestT, RequestResultT> requestSerializer )
        throws RequestSerializationException,
            RequestExecutionException,
            DestinationNotFoundException,
            DestinationAccessException
    {
        measurements.resetMeasurements();
        measurements.setBeginTotal(System.nanoTime());

        try {
            final SerializedRequest<RequestT> serializedRequest = serializeRequest(request, requestSerializer);
            final String responseBody = execute(destination, serializedRequest);
            return deserializeRequest(request, requestSerializer, responseBody);
        }
        finally {
            measurements.setEndTotal(System.nanoTime());
        }
    }

    @Nonnull
    private SerializedRequest<RequestT> serializeRequest(
        @Nonnull final RequestT request,
        @Nonnull final RequestSerializer<RequestT, RequestResultT> requestSerializer )
        throws RequestSerializationException,
            DestinationNotFoundException,
            DestinationAccessException
    {
        final long beginBuildReq = System.nanoTime();
        try {
            return requestSerializer.serialize(request);
        }
        finally {
            final long endBuildReq = System.nanoTime();
            measurements.addBuildRequestDuration(Duration.ofNanos(endBuildReq - beginBuildReq));
        }
    }

    @Nonnull
    private RequestResultT deserializeRequest(
        @Nonnull final RequestT request,
        @Nonnull final RequestSerializer<RequestT, RequestResultT> requestSerializer,
        @Nonnull final String responseBody )
        throws RequestSerializationException,
            DestinationNotFoundException,
            DestinationAccessException
    {
        final long beginParseResp = System.nanoTime();
        try {
            final SerializedRequestResult<RequestT> serializedRequestResult =
                new SerializedRequestResult<>(request, responseBody);
            return requestSerializer.deserialize(serializedRequestResult);
        }
        finally {
            final long endParseResp = System.nanoTime();
            measurements.addParseResponseDuration(Duration.ofNanos(endParseResp - beginParseResp));
        }
    }

    @Nonnull
    private RequestMethod getRequestMethod( @Nonnull final SerializedRequest<RequestT> serializedRequest )
    {
        return serializedRequest.getRequestMethod();
    }

    /**
     * Get the request URI.
     * 
     * @param destination
     *            The destination which is used for the HTTP request.
     * @param serializedRequest
     *            The serialized request payload.
     * @return The target request URI.
     */
    @Nonnull
    protected URI getRequestUri(
        @Nonnull final HttpDestination destination,
        @Nonnull final SerializedRequest<RequestT> serializedRequest )
    {
        return new ServiceUriBuilder().build(destination.getUri(), serializedRequest.getRequestPath());
    }

    @Nonnull
    private List<Header> getRequestHeaders(
        final ErpHttpDestination destination,
        @Nonnull final SerializedRequest<RequestT> serializedRequest )
    {
        final List<Header> requestHeaders = Lists.newArrayList(serializedRequest.getRequestHeaders());

        if( destination.getSapClient().isDefined() ) {
            // add HTTP header "sap-client" if a SAP client is defined
            final SapClient sapClient = destination.getSapClient().get();
            if( !sapClient.isDefault() && !sapClient.isEmpty() ) {
                requestHeaders.add(new Header(ErpHttpDestination.SAP_CLIENT_KEY, sapClient.getValue()));
            }
        }

        // add HTTP header "sap-language" if a language is defined
        final Locale locale = destination.getLocale();
        if( !Strings.isNullOrEmpty(locale.getLanguage()) ) {
            requestHeaders.add(new Header(ErpHttpDestination.LOCALE_KEY, locale.getLanguage()));
        }

        return requestHeaders;
    }

    private HttpUriRequest newRequest( final RequestMethod requestMethod, final URI requestUri )
    {
        switch( requestMethod ) {
            case GET:
                return new HttpGet(requestUri);
            case HEAD:
                return new HttpHead(requestUri);
            case POST:
                return new HttpPost(requestUri);
            case PUT:
                return new HttpPut(requestUri);
            case PATCH:
                return new HttpPatch(requestUri);
            case DELETE:
                return new HttpDelete(requestUri);
            case OPTIONS:
                return new HttpOptions(requestUri);
            default:
                throw new ShouldNotHappenException("Unsupported request method: " + requestMethod + ".");
        }
    }

    private HttpUriRequest newRequest(
        @Nonnull final RequestMethod requestMethod,
        @Nonnull final URI requestUri,
        @Nonnull final RequestBodyWithHeader bodyWithHeader )
        throws RequestSerializationException
    {
        final long beginBuildRequest = System.nanoTime();
        try {
            final HttpUriRequest request = newRequest(requestMethod, requestUri);

            if( request instanceof HttpEntityEnclosingRequest ) {
                ((HttpEntityEnclosingRequest) request).setEntity(getBodyAsCompressedEntity(bodyWithHeader.body));
            }

            request.setHeader(HttpHeaders.USER_AGENT, "sap-cloud-sdk");
            request.setHeader(HttpHeaders.ACCEPT_ENCODING, "gzip");

            for( final Header header : bodyWithHeader.headers ) {
                request.setHeader(header.getName(), header.getValue());
            }

            if( log.isTraceEnabled() ) {
                final Thread currentThread = Thread.currentThread();

                log.trace(
                    "Successfully prepared HTTP request for request execution (thread: "
                        + currentThread
                        + ", threat id: "
                        + currentThread.getId()
                        + ") URI: "
                        + requestUri
                        + " Body: "
                        + bodyWithHeader.body
                        + " Headers: "
                        + getNonSensitiveHeadersAsString(bodyWithHeader.headers)
                        + ".");
            }

            return request;
        }
        finally {
            measurements.addBuildRequestDuration(Duration.ofNanos(System.nanoTime() - beginBuildRequest));
        }
    }

    private void logReadAccessAttempt( final Request<?, ?> request, final ErpHttpDestination destination )
    {
        @Nullable
        final String readAccessData = request.getReadAccessData();

        if( readAccessData != null ) {
            AuditLogger.logDataReadAttempt(
                new AuditedDataObject(request.getClass().getSimpleName()),
                new AuditedDataSubject(
                    destination.get("Name", String.class::cast).getOrElse("Unnamed Destination"),
                    destination.getSapClient().get().getValue()),
                new AccessedAttribute(readAccessData, AccessedAttribute.Operation.READ));
        }
    }

    private String getRequestExecutionFailedMessage( final Request<?, ?> request )
    {
        return request.getClass().getSimpleName()
            + " "
            + request.getConstructedByMethod()
            + " failed ["
            + measurements.getMeasurementsString()
            + "]";
    }

    /**
     * Executes the given {@code serializedRequest} as a {@code HttpUriRequest}, returning the body of the
     * {@code HttpResponse} received.
     *
     * @param destination
     *            The {@code HttpDestination} of this call.
     * @param serializedRequest
     *            The {@code SerializedRequest} to execute.
     *
     * @return The body of the response received by the given request.
     *
     * @throws RequestSerializationException
     *             If the request could not be serialized.
     * @throws RequestExecutionException
     *             If any Exception occured during execution of the request.
     * @throws DestinationNotFoundException
     *             If the Destination cannot be found.
     * @throws DestinationAccessException
     *             If the destination is not of type DestinationType.HTTP or there is an issue while accessing
     *             destination information.
     */
    @Nonnull
    public String execute(
        @Nonnull final HttpDestination destination,
        @Nonnull final SerializedRequest<RequestT> serializedRequest )
        throws RequestSerializationException,
            RequestExecutionException,
            DestinationNotFoundException,
            DestinationAccessException
    {
        final RequestT request = serializedRequest.getRequest();

        final HttpClient httpClient = HttpClientAccessor.getHttpClient(destination);

        final RequestMethod requestMethod = getRequestMethod(serializedRequest);
        final URI requestUri = getRequestUri(destination, serializedRequest);

        // resolve request body and header for potentially signed request body
        final RequestBodyWithHeader bodyWithHeader = getRequestBodyWithHeader(destination, serializedRequest);
        final HttpUriRequest uriRequest = newRequest(requestMethod, requestUri, bodyWithHeader);

        HttpResponse response;
        final List<Header> responseHeaders = new ArrayList<>();
        final String responseBody;

        final long beginExecute = System.nanoTime();
        try {
            if( log.isDebugEnabled() ) {
                log.debug(
                    "Executing "
                        + request.getClass().getSimpleName()
                        + " constructed by: "
                        + request.getConstructedByMethod()
                        + ".");
            }

            logReadAccessAttempt(request, destination.decorate(asErp()));
            response = httpClient.execute(uriRequest);

            for( final org.apache.http.Header header : response.getAllHeaders() ) {
                responseHeaders.add(new Header(header.getName(), header.getValue()));
            }

            responseBody = HttpEntityUtil.getResponseBody(response);
        }
        catch( final RequestSerializationException e ) {
            if( log.isDebugEnabled() ) {
                log.debug(getRequestExecutionFailedMessage(request), e);
            }
            throw e;
        }
        catch( final Exception e ) {
            final String message = getRequestExecutionFailedMessage(request);
            throw new RequestExecutionException(message, e);
        }
        finally {
            measurements.addExecuteRequestDuration(Duration.ofNanos(System.nanoTime() - beginExecute));
        }

        if( responseBody == null ) {
            throw new RequestExecutionException("Failed to execute request: no body returned in response.");
        }

        handleHttpStatus(
            destination.decorate(asErp()),
            response.getStatusLine().getStatusCode(),
            responseBody,
            responseHeaders);
        return responseBody;
    }

    /**
     * Returns a wrapper object which encapsulates the HTTP request body and headers. This method can be overridden to
     * manipulate the request before submitting, e.g. signing queries, adding timestamps.
     *
     * @param destination
     *            The {@code HttpDestination} of this call.
     * @param request
     *            The {@code Request} to be executed.
     * @return The request body with header.
     */
    @Nonnull
    protected RequestBodyWithHeader getRequestBodyWithHeader(
        final HttpDestination destination,
        @Nonnull final SerializedRequest<RequestT> request )
    {
        return new RequestBodyWithHeader(
            getRequestHeaders(destination.decorate(asErp()), request),
            request.getRequestBody());
    }

    private static class ErpAuthorization extends Authorization
    {
        ErpAuthorization( @Nonnull final String missingAuthorization )
        {
            super(missingAuthorization);
        }
    }

    /**
     * A helper class to wrap request body and headers.
     */
    @Value
    static final class RequestBodyWithHeader
    {
        private final List<Header> headers;
        private final String body;
    }
}
