/*
 * Copyright (c) 2023 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.connectivity;

import java.net.URI;
import java.security.KeyStore;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Locale;

import javax.annotation.Nonnull;

import com.sap.cloud.sdk.cloudplatform.connectivity.AuthenticationType;
import com.sap.cloud.sdk.cloudplatform.connectivity.DefaultHttpDestination;
import com.sap.cloud.sdk.cloudplatform.connectivity.Header;
import com.sap.cloud.sdk.cloudplatform.connectivity.HttpDestinationProperties;
import com.sap.cloud.sdk.cloudplatform.connectivity.ProxyConfiguration;
import com.sap.cloud.sdk.cloudplatform.connectivity.ProxyType;
import com.sap.cloud.sdk.cloudplatform.security.BasicCredentials;
import com.sap.cloud.sdk.s4hana.serialization.SapClient;

import io.vavr.control.Option;
import lombok.EqualsAndHashCode;
import lombok.extern.slf4j.Slf4j;

/**
 * Delegator implementation of the {@link ErpHttpDestination} interface, delegating all
 * {@link HttpDestinationProperties} calls to the wrapped destination.
 */
@EqualsAndHashCode
@Slf4j
public class DefaultErpHttpDestination implements ErpHttpDestination
{
    /**
     * The header which will get the stored SapClient set.
     */
    public static final String SAP_CLIENT_HEADER_NAME = "sap-client";

    /**
     * The header which will get the stored Locale set.
     */
    public static final String LOCALE_HEADER_NAME = "sap-language";

    private final HttpDestinationProperties baseDestination;

    /**
     * Constructor wrapping the given destination and redirecting all {@link HttpDestinationProperties} calls to the
     * given destination.
     * 
     * @param baseDestination
     *            The destination to wrap.
     */
    public DefaultErpHttpDestination( @Nonnull final HttpDestinationProperties baseDestination )
    {
        this.baseDestination = baseDestination;
    }

    @Override
    @Nonnull
    public Option<Object> get( @Nonnull final String key )
    {
        return baseDestination.get(key);
    }

    @Override
    @Nonnull
    public Iterable<String> getPropertyNames()
    {
        return baseDestination.getPropertyNames();
    }

    @Override
    @Nonnull
    public URI getUri()
    {
        return baseDestination.getUri();
    }

    @Override
    @Nonnull
    public Option<String> getTlsVersion()
    {
        return baseDestination.getTlsVersion();
    }

    @Override
    @Nonnull
    public Option<ProxyConfiguration> getProxyConfiguration()
    {
        return baseDestination.getProxyConfiguration();
    }

    @Override
    @Nonnull
    public Option<KeyStore> getKeyStore()
    {
        return baseDestination.getKeyStore();
    }

    @Override
    @Nonnull
    public Option<String> getKeyStorePassword()
    {
        return baseDestination.getKeyStorePassword();
    }

    @Override
    public boolean isTrustingAllCertificates()
    {
        return baseDestination.isTrustingAllCertificates();
    }

    @Override
    @Nonnull
    public Option<BasicCredentials> getBasicCredentials()
    {
        return baseDestination.getBasicCredentials();
    }

    @Override
    @Nonnull
    public Option<ProxyType> getProxyType()
    {
        return baseDestination.getProxyType();
    }

    /**
     * Get the destination name.
     * 
     * @return The destination name.
     */
    @Nonnull
    public Option<String> getName()
    {
        return baseDestination.get("Name", String.class::cast);
    }

    @Override
    @Nonnull
    public Option<KeyStore> getTrustStore()
    {
        return baseDestination.getTrustStore();
    }

    @Override
    @Nonnull
    public Option<String> getTrustStorePassword()
    {
        return baseDestination.getTrustStorePassword();
    }

    @Nonnull
    @Override
    public AuthenticationType getAuthenticationType()
    {
        return baseDestination.getAuthenticationType();
    }

    @Nonnull
    @Override
    public Collection<Header> getHeaders( @Nonnull final URI requestUri )
    {
        final io.vavr.collection.List<Header> providedHeaders =
            io.vavr.collection.List.ofAll(baseDestination.getHeaders(requestUri));

        final List<Header> headersToAdd = getHeadersToAdd();

        headersToAdd
            .forEach(
                header -> log.debug("Adding request header {} for HTTP destination pointing to {}.", header, getUri()));

        // the order in which those lists are combined determines which are kept in the end:
        // - appendAll: the additional headers are discarded if the header name is already in use
        // - prependAll: the additional headers replace the original header if the header name is already in use
        final io.vavr.collection.List<Header> allHeaders = providedHeaders.appendAll(headersToAdd);

        // distinctBy keeps the first occurrence of a header
        return allHeaders.distinctBy(Header::getName).toJavaList();
    }

    private List<Header> getHeadersToAdd()
    {
        final List<Header> result = new ArrayList<>();

        result.add(new Header(LOCALE_HEADER_NAME, getLocale().getLanguage()));

        final Option<SapClient> maybeSapClient = getSapClient();

        if( isOnPremiseDestinationWithoutSapClient(maybeSapClient) ) {
            log
                .info(
                    "No {} property defined on HTTP destination pointing to on-premise ERP system with URI {}. It is recommended to specify the {} property to prevent authentication issues.",
                    ErpHttpDestinationProperties.SAP_CLIENT_KEY,
                    getUri(),
                    ErpHttpDestinationProperties.SAP_CLIENT_KEY);
        }

        maybeSapClient.peek(sapClient -> result.add(new Header(SAP_CLIENT_HEADER_NAME, sapClient.getValue())));

        return result;
    }

    private boolean isOnPremiseDestinationWithoutSapClient( final Option<SapClient> maybeSapClient )
    {
        return maybeSapClient.isEmpty() && getProxyType().isDefined() && getProxyType().get() == ProxyType.ON_PREMISE;
    }

    /**
     * Starts a builder to be used to create a {@code DefaultErpHttpDestination} with some properties.
     *
     * @param uri
     *            The uri of the {@code DefaultErpHttpDestination} to be created.
     * @return A new {@code Builder} instance.
     */
    @Nonnull
    public static Builder builder( @Nonnull final URI uri )
    {
        return new Builder(uri);
    }

    /**
     * Starts a builder to be used to create a {@code DefaultErpHttpDestination} with some properties.
     *
     * @param uri
     *            The uri of the {@code DefaultErpHttpDestination} to be created. In case this is no valid URI an
     *            {@link IllegalArgumentException} is thrown.
     * @return A new {@code Builder} instance.
     * @throws IllegalArgumentException
     *             if the given {@code uri} is no valid URI.
     */
    @Nonnull
    public static Builder builder( @Nonnull final String uri )
    {
        return builder(URI.create(uri));
    }

    /**
     * Builder class to allow for easy creation of an immutable {@code DefaultErpHttpDestination} instance.
     */
    public static class Builder
    {
        private final DefaultHttpDestination.Builder builder;

        Builder( @Nonnull final URI uri )
        {
            builder = DefaultHttpDestination.builder(uri);
        }

        /**
         * Sets the sap client to be used by the destination currently build.
         * 
         * @param sapClient
         *            The sap client to use.
         * @return This builder.
         */
        @Nonnull
        public Builder sapClient( @Nonnull final SapClient sapClient )
        {
            builder.property(SAP_CLIENT_KEY, sapClient.getValue());
            return this;
        }

        /**
         * Sets the locale to be used by the destination currently build.
         * 
         * @param locale
         *            The locale to use.
         * @return This builder.
         */
        @Nonnull
        public Builder locale( @Nonnull final Locale locale )
        {
            builder.property(LOCALE_KEY, locale.getLanguage());
            return this;
        }

        /**
         * Sets the TLS version used by the {@code DefaultErpHttpDestination} to the given value.
         *
         * @param value
         *            The TLS version that should be used.
         * @return This builder.
         */
        @Nonnull
        public Builder tlsVersion( @Nonnull final String value )
        {
            builder.tlsVersion(value);
            return this;
        }

        /**
         * Adds the given key-value pair to the destination to be created. This will overwrite any property already
         * assigned to the key.
         *
         * @param key
         *            The key to assign a property for.
         * @param value
         *            The property value to be assigned.
         * @return This builder.
         */
        @Nonnull
        public Builder property( @Nonnull final String key, @Nonnull final Object value )
        {
            builder.property(key, value);
            return this;
        }

        /**
         * Sets the key store password for the corresponding {@link KeyStore} used by the
         * {@link DefaultErpHttpDestination} to access the key store.
         *
         * @param value
         *            The keyStore password that should be used.
         * @return This builder.
         *
         */
        @Nonnull
        public Builder keyStorePassword( @Nonnull final String value )
        {
            builder.keyStorePassword(value);
            return this;
        }

        /**
         * Sets the {@link KeyStore} to be used when communicating over HTTP.
         *
         * @param keyStore
         *            The keyStore that should be used. for HTTP communication
         * @return This builder.
         *
         */
        @Nonnull
        public Builder keyStore( @Nonnull final KeyStore keyStore )
        {
            builder.keyStore(keyStore);
            return this;
        }

        /**
         * Lets the {@code DefaultErpHttpDestination} trust all server certificates.
         *
         * @return This builder.
         */
        @Nonnull
        public Builder trustAllCertificates()
        {
            builder.trustAllCertificates();
            return this;
        }

        /**
         * Sets the name of the {@code DefaultErpHttpDestination}.
         *
         * @param name
         *            The destination name
         * @return This builder.
         */
        @Nonnull
        public Builder name( @Nonnull final String name )
        {
            builder.name(name);
            return this;
        }

        /**
         * Sets the proxy URI of the {@code DefaultErpHttpDestination}.
         *
         * @param proxyUri
         *            The URI of the proxy
         * @return This builder.
         */
        @Nonnull
        public Builder proxy( @Nonnull final URI proxyUri )
        {
            builder.proxy(proxyUri);
            return this;
        }

        /**
         * Sets the proxy host and proxy port of the {@code DefaultErpHttpDestination}.
         *
         * @param proxyHost
         *            The host of the proxy
         * @param proxyPort
         *            The port of the proxy
         * @return This builder.
         */
        @Nonnull
        public Builder proxy( @Nonnull final String proxyHost, final int proxyPort )
        {
            builder.proxy(proxyHost, proxyPort);
            return this;
        }

        /**
         * Sets the proxy type (Internet or On-Premise).
         *
         * @param proxyType
         *            Type of proxy this destination is configured for.
         * @return This builder.
         */
        @Nonnull
        public Builder proxyType( @Nonnull final ProxyType proxyType )
        {
            return property("proxyType", proxyType);
        }

        /**
         * Sets the enum value for {@link AuthenticationType}.
         *
         * @param authenticationType
         *            The authentication type used for the destination
         * @return This builder.
         */
        @Nonnull
        public Builder authenticationType( @Nonnull final AuthenticationType authenticationType )
        {
            builder.authenticationType(authenticationType);
            return this;
        }

        /**
         * Sets the user name of the {@code DefaultErpHttpDestination}.
         *
         * @param user
         *            The user name of the destination
         * @return This builder.
         */
        @Nonnull
        public Builder user( @Nonnull final String user )
        {
            builder.user(user);
            return this;
        }

        /**
         * Sets the password of the {@code DefaultErpHttpDestination}.
         *
         * @param password
         *            The password of the destination
         * @return This builder.
         */
        @Nonnull
        public Builder password( @Nonnull final String password )
        {
            builder.password(password);
            return this;
        }

        /**
         * Sets the credentials for accessing the destination when basic authentication is used.
         *
         * @param basicCredentials
         *            Username and password represented as a {@link BasicCredentials} object.
         * @return This builder.
         **/
        @Nonnull
        public Builder basicCredentials( @Nonnull final BasicCredentials basicCredentials )
        {
            if( basicCredentials != null ) {
                builder.user(basicCredentials.getUsername());
                builder.password(basicCredentials.getPassword());
            }
            return this;
        }

        /**
         * Adds the given header to the list of headers added to every outgoing request for this destination.
         *
         * @param header
         *            A header to add to outgoing requests.
         * @return This builder.
         */
        @Nonnull
        public Builder header( @Nonnull final Header header )
        {
            builder.header(header);
            return this;
        }

        /**
         * Adds a header given by the {@code headerName} and {@code headerValue} to the list of headers added to every
         * outgoing request for this destination.
         *
         * @param headerName
         *            The name of the header to add.
         * @param headerValue
         *            The value of the header to add.
         * @return This builder.
         */
        @Nonnull
        public Builder header( @Nonnull final String headerName, @Nonnull final String headerValue )
        {
            builder.header(headerName, headerValue);
            return this;
        }

        /**
         * Finally creates the {@code DefaultErpHttpDestination} with the properties retrieved via the
         * {@link #property(String, Object)} method.
         *
         * @return A fully instantiated {@code DefaultErpHttpDestination}.
         */
        @Nonnull
        public DefaultErpHttpDestination build()
        {
            return new DefaultErpHttpDestination(builder.build());
        }
    }
}
