/*
 * Copyright (c) 2023 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.s4hana.connectivity;

import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.util.Arrays;
import java.util.Collection;
import java.util.Map;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import com.google.common.collect.Maps;
import com.sap.cloud.sdk.cloudplatform.exception.ShouldNotHappenException;
import com.sap.cloud.sdk.s4hana.serialization.BigDecimalConverter;
import com.sap.cloud.sdk.s4hana.serialization.BigIntegerConverter;
import com.sap.cloud.sdk.s4hana.serialization.BooleanConverter;
import com.sap.cloud.sdk.s4hana.serialization.ByteConverter;
import com.sap.cloud.sdk.s4hana.serialization.CharacterConverter;
import com.sap.cloud.sdk.s4hana.serialization.DoubleConverter;
import com.sap.cloud.sdk.s4hana.serialization.ErpBooleanConverter;
import com.sap.cloud.sdk.s4hana.serialization.ErpDecimalConverter;
import com.sap.cloud.sdk.s4hana.serialization.ErpType;
import com.sap.cloud.sdk.s4hana.serialization.ErpTypeConverter;
import com.sap.cloud.sdk.s4hana.serialization.FloatConverter;
import com.sap.cloud.sdk.s4hana.serialization.IntegerConverter;
import com.sap.cloud.sdk.s4hana.serialization.LocalDateConverter;
import com.sap.cloud.sdk.s4hana.serialization.LocalTimeConverter;
import com.sap.cloud.sdk.s4hana.serialization.LocaleConverter;
import com.sap.cloud.sdk.s4hana.serialization.LongConverter;
import com.sap.cloud.sdk.s4hana.serialization.ShortConverter;
import com.sap.cloud.sdk.s4hana.serialization.YearConverter;
import com.sap.cloud.sdk.typeconverter.ConvertedObject;

import lombok.extern.slf4j.Slf4j;

/**
 * Used for serialization and deserialization of ERP-based types.
 */
@Slf4j
public class ErpTypeSerializer
{
    private final Map<Class<?>, ErpTypeConverter<?>> typeConverters = Maps.newIdentityHashMap();

    /**
     * Initializes the {@link ErpTypeSerializer} with default ERP type converters.
     */
    public ErpTypeSerializer()
    {
        withTypeConverters(
            BooleanConverter.INSTANCE,
            CharacterConverter.INSTANCE,
            ByteConverter.INSTANCE,
            ShortConverter.INSTANCE,
            IntegerConverter.INSTANCE,
            LongConverter.INSTANCE,
            FloatConverter.INSTANCE,
            DoubleConverter.INSTANCE,
            BigIntegerConverter.INSTANCE,
            BigDecimalConverter.INSTANCE,
            YearConverter.INSTANCE,
            LocalDateConverter.INSTANCE,
            LocalTimeConverter.INSTANCE,
            LocaleConverter.INSTANCE,
            ErpDecimalConverter.INSTANCE,
            ErpBooleanConverter.INSTANCE);
    }

    /**
     * Registers the given {@link ErpTypeConverter}s. Replaces existing converters for already existing types that have
     * been added before.
     * 
     * @param typeConverters
     *            The ERP type converters to be added.
     * @return The same instance.
     */
    @Nonnull
    public ErpTypeSerializer withTypeConverters( @Nonnull final Iterable<ErpTypeConverter<?>> typeConverters )
    {
        for( final ErpTypeConverter<?> typeConverter : typeConverters ) {
            this.typeConverters.put(typeConverter.getType(), typeConverter);
        }

        return this;
    }

    /**
     * Delegates to {@link #withTypeConverters(Iterable)}.
     *
     * @param typeConverters
     *            The ERP type converters to be added.
     * @return The same instance.
     */
    @Nonnull
    public ErpTypeSerializer withTypeConverters( @Nonnull final ErpTypeConverter<?>... typeConverters )
    {
        return withTypeConverters(Arrays.asList(typeConverters));
    }

    /**
     * Gets registered {@link ErpTypeConverter}s.
     *
     * @return All registered ERP type converters.
     */
    @Nonnull
    public Collection<ErpTypeConverter<?>> getTypeConverters()
    {
        return typeConverters.values();
    }

    /**
     * Gets registered {@link ErpTypeConverter}s for each {@link ErpType}.
     *
     * @return The converters for each ERP type.
     */
    @Nonnull
    public Map<Class<?>, ErpTypeConverter<?>> getTypeConvertersByType()
    {
        return typeConverters;
    }

    @SuppressWarnings( "unchecked" )
    @Nullable
    private <T> ErpTypeConverter<T> getTypeConverter( final Class<T> type )
    {
        return (ErpTypeConverter<T>) typeConverters.get(type);
    }

    /**
     * Convert given object to an ERP type using a registered {@link ErpTypeConverter}.
     * <p>
     * Example usage in the SDK {@link ErpTypeSerializer} test:
     * 
     * <pre>
     * {@code}
     * final ErpTypeSerializer serializer = new ErpTypeSerializer();
     * assertThat(serializer.toErp(new CostCenter("123")).get()).isEqualTo("0000000123");
     * assertThat(serializer.toErp(-123.4d).get()).isEqualTo("123.4-");
     * </pre>
     *
     * @param object
     *            The ERP object to serialize.
     * @param <T>
     *            The generic type.
     * @return A wrapped instance of the serialized object.
     */
    @Nonnull
    public <T> ConvertedObject<String> toErp( @Nullable final T object )
    {
        if( object == null ) {
            return ConvertedObject.ofNull();
        }

        final ConvertedObject<String> erpObject;

        if( object instanceof ErpType<?> ) {
            @SuppressWarnings( "unchecked" )
            final ErpTypeConverter<T> converter = (ErpTypeConverter<T>) ((ErpType<?>) object).getTypeConverter();

            erpObject = converter.toDomain(object);
        } else {
            @SuppressWarnings( "unchecked" )
            final ErpTypeConverter<T> converter = (ErpTypeConverter<T>) getTypeConverter(object.getClass());

            if( converter != null ) {
                erpObject = converter.toDomain(object);
            } else {
                erpObject = ConvertedObject.of(object.toString());
            }
        }

        return erpObject;
    }

    /**
     * Convert a given String based erpObject in the ERP-based representation into an object of resultType. For the
     * conversion, uses a {@link ErpTypeConverter} registered for resultType.
     * <p>
     * * Example usage in the SDK {@link ErpTypeSerializer} test:
     * 
     * <pre>
     * {@code}
     * assertThat(serializer.fromErp("0000000123", CostCenter.class).get()).isEqualTo(new CostCenter("123"));
     * assertThat(serializer.fromErp("123.4 ", Double.class).get()).isEqualTo(123.4d);
     * assertThat(serializer.fromErp("123.4-", Double.class).get()).isEqualTo(-123.4d);
     * </pre>
     *
     * @param erpObject
     *            The serialized ERP object.
     * @param resultType
     *            The expected deserialization result type.
     * @param <T>
     *            The generic result type.
     * @return A wrapped instance of the deserialized object.
     */
    @Nonnull
    public <T> ConvertedObject<T> fromErp( @Nullable final String erpObject, @Nonnull final Class<T> resultType )
    {
        if( erpObject == null ) {
            return ConvertedObject.ofNull();
        }

        final ErpTypeConverter<T> converter = getTypeConverter(resultType);

        if( converter != null ) {
            return converter.fromDomain(erpObject);
        } else {
            try {
                final Constructor<T> stringConstructor = resultType.getConstructor(String.class);
                return ConvertedObject.of(stringConstructor.newInstance(erpObject));
            }
            catch( final
                NoSuchMethodException
                    | SecurityException
                    | IllegalAccessException
                    | InstantiationException e ) {
                throw new ShouldNotHappenException(
                    String
                        .format(
                            "Failed to instantiate object from %s: No constructor available with %s parameter.",
                            resultType.getSimpleName(),
                            String.class.getSimpleName()),
                    e);
            }
            catch( final InvocationTargetException e ) {
                if( log.isDebugEnabled() ) {
                    log.debug("Failed to convert ERP object to " + resultType.getName() + ": " + erpObject + ".");
                }
                return ConvertedObject.ofNotConvertible();
            }
        }
    }
}
