/*******************************************************************************
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements. See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership. The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations
 * under the License.
 ******************************************************************************/
package org.apache.olingo.odata2.client.api.ep;

import java.net.URI;

/**
 * {@link EntitySerializerProperties} contains all additional properties which are necessary to <b>write
 * (serialize)</b> an {@link org.apache.olingo.odata2.api.ep.entry.ODataEntry} into an specific format (e.g.
 * <code>XML</code> or <code>JSON</code> or ...).
 */
public class EntitySerializerProperties {

  private URI serviceRoot;
  private boolean includeMetadata = false;
  private boolean validatingFacets = true;
  private boolean isKeyAutoGenerated = false;
  private boolean isDateTimeJsonRelaxed = false;
  
  private EntitySerializerProperties() {}
  
  /**
   * Set true or false to include metadata
   * @return boolean
   */
  public boolean isIncludeMetadata() {
    return includeMetadata;
  }

  /**
   * Set true or false for auto key generation
   * @return boolean
   */
  public final boolean isKeyAutoGenerated() {
    return isKeyAutoGenerated;
  }

  /**
   * Set true or false if datetime value 
   * for json has to be relaxed
   * @return boolean
   */
  public final boolean isDateTimeJsonRelaxed() {
    return isDateTimeJsonRelaxed;
  }
  
  /**
   * Gets the service root.
   * @return the service root
   */
  public final URI getServiceRoot() {
    return serviceRoot;
  }

  /**
   * 
   * @param serviceRoot
   * @return EntitySerializerPropertiesBuilder
   */
  public static EntitySerializerPropertiesBuilder serviceRoot(final URI serviceRoot) {
    return new EntitySerializerPropertiesBuilder().serviceRoot(serviceRoot);
  }

  /**
   * 
   * @return boolean
   */
  public boolean isValidatingFacets() {
    return validatingFacets;
  }

  /**
   * This class builds the Entity Serializer Properties
   *
   */
  public static class EntitySerializerPropertiesBuilder {
    private final EntitySerializerProperties properties = new EntitySerializerProperties();

    /**
     * 
     * @param includeMetadata
     * @return EntitySerializerPropertiesBuilder
     */
    public EntitySerializerPropertiesBuilder
      includeMetadata(final boolean includeMetadata) {
    properties.includeMetadata = includeMetadata;
    return this;
   }

   

    /**
     * @param serviceRoot
     */
    public final EntitySerializerPropertiesBuilder serviceRoot(final URI serviceRoot) {
      properties.serviceRoot = serviceRoot;
      return this;
    }
    
    /**
     * @param setting if payload has auto generated keys
     */
    public final EntitySerializerPropertiesBuilder isKeyAutoGenerated
    (boolean isKeyAutoGenerated) {
      properties.isKeyAutoGenerated = isKeyAutoGenerated;
      return this;
    }
    
    /**
     * @param setting if datetime value in payload has to be relaxed
     */
    public final EntitySerializerPropertiesBuilder isDateTimeJsonRelaxed
    (boolean isDateTimeJsonRelaxed) {
      properties.isDateTimeJsonRelaxed = isDateTimeJsonRelaxed;
      return this;
    }

    /**
     * Build properties object.
     * @return assembled properties object
     */
    public final EntitySerializerProperties build() {
      return properties;
    }

    /**
     * 
     * @param validatingFacets
     * @return EntitySerializerPropertiesBuilder
     */
    public EntitySerializerPropertiesBuilder validatingFacets(final boolean validatingFacets) {
      properties.validatingFacets = validatingFacets;
      return this;
    }

    /**
     * 
     * @param properties
     * @return EntitySerializerPropertiesBuilder
     */
    public EntitySerializerPropertiesBuilder fromProperties
    (final EntitySerializerProperties properties) {
      this.properties.validatingFacets = properties.validatingFacets;
      this.properties.includeMetadata = properties.includeMetadata;
      this.properties.isKeyAutoGenerated = properties.isKeyAutoGenerated;
      return this;
    }

  }

  /**
   * 
   * @param properties
   * @return EntitySerializerPropertiesBuilder
   */
  public static EntitySerializerPropertiesBuilder fromProperties
  (final EntitySerializerProperties properties) {
    final EntitySerializerPropertiesBuilder builder =
        EntitySerializerProperties.serviceRoot(properties.getServiceRoot());
    return builder.fromProperties(properties);
  }
}
