/*******************************************************************************
 * (c) 201X SAP SE or an SAP affiliate company. All rights reserved.
 ******************************************************************************/
package com.sap.cloud.sdk.odatav2.connectivity;

import java.util.HashMap;

import org.apache.olingo.odata2.api.edm.EdmEntityType;
import org.apache.olingo.odata2.api.edm.EdmException;
import org.apache.olingo.odata2.api.edm.EdmNavigationProperty;
import org.apache.olingo.odata2.api.edm.EdmType;
import org.apache.olingo.odata2.api.edm.EdmTypeKind;
import org.apache.olingo.odata2.api.uri.expression.MethodOperator;

import com.sap.cloud.sdk.odatav2.connectivity.filter.FilterHelperUtil;
import com.sap.cloud.sdk.odatav2.connectivity.internal.UrlEncoder;

import lombok.RequiredArgsConstructor;

@RequiredArgsConstructor
public class FilterExpression {
	private static final String BRACKET_OPEN = "(";
	private static final String BRACKET_CLOSE = ")";
	private HashMap<String, EdmType> typeMap = new HashMap<>();
	private final String AND = "and";
	private final String OR = "or";
	private final Object left;
	private final String operator;
	private final ODataType right;
	private FilterExpression rightFilter = null;
	private boolean isNot = false;
	private boolean isLeftFilterFunction = false;
	
	public static FilterExpression not(FilterExpression filter) {
		filter.setNot();
		return filter;
	}

	/**
	 * Creates a simple filter expression of the form field operator operand
	 * @param field Name of the property/field 
	 * @param operator operator as a String. For example:- eq, gt.
	 * @param right an ODataType instance representing the operand.
	 */
	public FilterExpression(String field, String operator, ODataType right) {
		left = field;
		this.operator = operator;
		this.right = right;
		rightFilter = null;
	}
	
	/**
	 * Creates a simple filter expression of the form field operator operand
	 * @param field Name of the property/field 
	 * @param operator operator as a String. For example:- eq, gt.
	 * @param right an ODataType instance representing the operand.
	 * @param isLeftFilterFunction specifies if left operand is filter function
	 */
	public FilterExpression(String field, String operator, ODataType right, boolean isLeftFilterFunction) {
		left = field;
		this.operator = operator;
		this.right = right;
		rightFilter = null;
		this.isLeftFilterFunction = isLeftFilterFunction;
	}

	private FilterExpression(FilterExpression left, String operator, FilterExpression right) {
		this.left = left;
		this.rightFilter = right;
		this.operator = operator;
		this.right = null;
	}

	/**
	 * Returns a filterExpression which is the OR of this filterExpression and the parameter filterExpression.
	 * Note that this forms a single filter expression resulting from the OR operation of 2 filter expressions. This filter expression is a single unit,
	 * ie there is an "implicit parenthesis" around the filter expression. For example:-
	 * 
	 * FilterExpression AB = A.or(B); // This is not just A or B but (A or B).
	 * 
	 * @param orExpression is the filterExpression which will be ORed with this filterExpression.
	 * 
	 */
	public FilterExpression or(final FilterExpression orExpression) {
		return new FilterExpression(this, OR, orExpression);
	}

	/**
	 * Returns a filterExpression which is the AND of this filterExpressions and the parameter filterExpression.
	 * Note that this forms a single filter expression resulting from the AND operation of 2 filter expressions. This filter expression is a single unit,
	 * ie there is an "implicit parenthesis" around the filter expression. For example:-
	 * 
	 * FilterExpression AB = A.and(B); // This is not just A and B but (A and B).
	 * 
	 * @param andExpression is the filterExpression which will be ANDed with this filterExpression.
	 * 
	 */
	public FilterExpression and(final FilterExpression andExpression) {
		return new FilterExpression(this, AND, andExpression);
	}

	private String toStringBasic(final EdmEntityType entityType, boolean isFilterFunction) throws EdmException {
		String converted = null;
		String field = left.toString();
		ODataType value = (ODataType) right;
		if(value != null){
		// in case there is metadata available
			if (entityType != null) {
				saveType(field, entityType);
				if (typeMap.containsKey(field)) {
					Object val = value.getValue();
					if(val instanceof String){
						val = UrlEncoder.encode(val.toString());
					}
					converted = ODataTypeValueSerializer.of(typeMap.get(field)).toUri(val);
				}
			}
	
			// in case there is NO metadata available
			else {
				// use default String representation for all types
				String stringVal = value.getValue().toString();
				if(value.getValue() instanceof String)
					stringVal = "'" + stringVal + "'";
				converted = isFilterFunction ? stringVal : UrlEncoder.encode(stringVal);
	
			}
		}	
		String filterString = field + " " + operator + " " + converted;
		return prependNOTIfRequired(filterString);
	}
	
	private void saveType(String property, EdmEntityType entityType) throws EdmException {
		if (typeMap.containsKey(property))
			return;
		EdmEntityType targetEntityType = entityType;
		String targetProperty;
		
		/*
		 * The below code finds out the type of the property, in which property can be a direct property
		 * like ProductID or can be a property of a child entity like "Category/CategoryName"
		 * we handle both the conditions with the for loop below. (The for loop does not execute when
		 * it is a direct property, and in the second case, we iterate thru the navigation path,
		 * and find out the target entity type of the child entity which the property belongs to.
		 * Once  the target entity type is obtained we get the Edm type in straightforward manner.
		 */
		String[] navigationPath = property.split("/");
		int pathLengthExcludingProperty = navigationPath.length - 1;
		targetProperty = navigationPath[pathLengthExcludingProperty];
		for (int i = 0;i<pathLengthExcludingProperty;i++) {
			String navigationProperty = navigationPath[i];
				targetEntityType = getTargetEntityTypeFromNavigationProperty(targetEntityType, navigationProperty);
		}
		if(targetEntityType.getProperty(targetProperty) != null && targetEntityType.getProperty(targetProperty).getType().getKind().equals(EdmTypeKind.SIMPLE))
			typeMap.put(property, targetEntityType.getProperty(targetProperty).getType());
			
	}

	private EdmEntityType getTargetEntityTypeFromNavigationProperty(EdmEntityType entityType,
			String navigationProperty) throws EdmException {
		EdmNavigationProperty navProp = (EdmNavigationProperty) entityType.getProperty(navigationProperty);
		String toRole = navProp.getToRole();
		return (EdmEntityType) navProp.getRelationship().getEnd(toRole).getEntityType();
	}

	public String toString(final EdmEntityType entityType) throws EdmException {
		if(isMethodOperator()){		
			return getFilterString();
		}
		if (!(left instanceof FilterExpression) && !this.isLeftFilterFunction) {
			return toStringBasic(entityType, this.isLeftFilterFunction);
		}
		else if(!(left instanceof FilterExpression) && this.isLeftFilterFunction) {
			return toStringBasic(null, this.isLeftFilterFunction);
		}
		FilterExpression leftFilter = (FilterExpression) left;
		String rightExp = rightFilter != null ? rightFilter.toString(entityType):right.toString();
		String filterString = BRACKET_OPEN + leftFilter.toString(entityType) + BRACKET_CLOSE +" " + operator + " " + BRACKET_OPEN + rightExp + BRACKET_CLOSE;
		
		return prependNOTIfRequired(filterString);
	}

	private String getFilterString() throws EdmException {
		MethodOperator method = MethodOperator.valueOf(operator.toUpperCase());
		switch(method){
			case SUBSTRINGOF:
				return FilterHelperUtil.buildSubstringOfFunc(this.operator.toString(), this.right, this.left.toString());
			case STARTSWITH:
			case ENDSWITH:	
				return FilterHelperUtil.buildBinaryFunc(this.operator.toString(), this.left.toString(), this.right);
			default:
				throw new EdmException(null, "FILTER_NOT_SUPPORTED");
		}
	}

	private boolean isMethodOperator() {
		try{
			MethodOperator.valueOf(operator.toUpperCase());
			return true;
		}catch(IllegalArgumentException e){
			return false;
		}
	}

	@Override
	public String toString() {
		try {
			return toString(null);
		} catch (EdmException e) {
			return e.getMessage(); // This is sufficient since this method is used for testing and debugging purposes only.
		}

	}
	
	private void setNot() {
		isNot = true;
	}
	
	private String prependNOTIfRequired(String filterString) {
		if(isNot)
			return "not (" + filterString + BRACKET_CLOSE;
		return filterString;
	}

}
