/*******************************************************************************
 * (c) 201X SAP SE or an SAP affiliate company. All rights reserved.
 ******************************************************************************/
package com.sap.cloud.sdk.odatav2.connectivity;

import java.net.URL;
import java.util.HashMap;
import java.util.Map;

import com.sap.cloud.sdk.cloudplatform.cache.CacheKey;
import com.sap.cloud.sdk.odatav2.connectivity.impl.ODataDeleteRequestImpl;

public class ODataDeleteRequestBuilder {

	private String serviceName;
	private String entitySetName;
	private Map<String, Object> keys;
	private ErrorResultHandler<?> errorHandler;
	private Map<String, String> headers = new HashMap<String, String>();
	private Map<String, String> destinationRelevantHeaders = new HashMap<String, String>();
	private boolean cacheMetadata = false;
	private URL metadataFilePath;
	private CacheKey cacheKey;
	private boolean isCacheRefresh = false;

	private ODataDeleteRequestBuilder(String serviceName, String entitySetName, Map<String, Object> keys) {
		this.serviceName = serviceName;
		this.entitySetName = entitySetName;
		this.keys = keys;
	}

	/**
	 * Sets an error handler to this ODataDeleteRequestBuilder.
	 * 
	 * @param errorHandler
	 * @return ODataCreateRequestBuilder
	 */
	public ODataDeleteRequestBuilder withErrorHandler(ErrorResultHandler<?> errorHandler) {
		this.errorHandler = errorHandler;
		return this;
	}

	/**
     * Enables caching of the metadata of an OData V2 data source. If your application is running on a tenant, then the tenant ID along with the metadata URL is used to form the cache key.
     * @return ODataDeleteRequestBuilder
     */
	public ODataDeleteRequestBuilder enableMetadataCache() {
  	  	this.cacheMetadata = true;
  	  return this;
    }
	
	 /**
     * Enables caching of the metadata of an OData V2 data source.
     * @param key {@link com.sap.cloud.sdk.cloudplatform.cache.CacheKey Cache key} containing the ID of the tenant where the application runs. You can also include the user name in the cache key.
     * @return ODataDeleteRequestBuilder
	 */
	public ODataDeleteRequestBuilder enableMetadataCache(CacheKey cacheKey){
    	this.cacheKey = cacheKey;
    	this.cacheMetadata = true;
		return this;
    }
	
	 /**
	   * Replaces the existing metadata in the cache with the latest version from the OData V2 data source.
	   * @return ODataDeleteRequestBuilder
	   */
	public ODataDeleteRequestBuilder withCacheRefresh(){
    	this.isCacheRefresh = true;
    	return this;
    	
    }
	
	/**
	   * Gets the metadata from the specified path.
	   * @param metadataFilePath URL pointing to the metadata information
	   * @return ODataDeleteRequestBuilder A builder for forming the Create
	   */
	public ODataDeleteRequestBuilder withMetadata(URL metadataFilePath){
    	this.metadataFilePath = metadataFilePath;
    	return this;
    }

	/**
	 * Creates an ODataDeleteRequestBuilder with the given service and entity name.
	 * 
	 * @param serviceName
	 *            name of the odata service where the developer wants to execute a
	 *            delete operation.
	 * @param entitySetName
	 *            name of the entity set, on which the developer wants to execute a
	 *            delete operation
	 * @param keys
	 *            keys identifying a particular entity
	 * @return ODataDeleteRequestBuilder
	 */
	public static ODataDeleteRequestBuilder withEntity(String serviceName, String entitySetName,
			Map<String, Object> keys) {
		return new ODataDeleteRequestBuilder(serviceName, entitySetName, keys);
	}

	public ODataDeleteRequestBuilder withHeader(String key, String value) {
		return withHeader(key, value, false);
	}

	/**
	 * Adds a header to the delete request and optionally to the metadata request as
	 * well depending on the value of the passInAllRequests parameter.
	 * 
	 * @param key
	 *            name of the header
	 * @param value
	 *            value of the header
	 * @param passInAllRequests
	 *            boolean indicating whether the header is to be passed in all the
	 *            requests to the backend like $metadata call, CSRF Token fetch etc. made as part of the Delete Request call
	 * @return ODataQueryBuilder
	 */
	public ODataDeleteRequestBuilder withHeader(String key, String value, boolean passInAllRequests) {

		if (passInAllRequests)
			destinationRelevantHeaders.put(key, value);// These headers are added to the metadata request.
		
		if(key.equals("SAP-PASSPORT") && !passInAllRequests)
	          destinationRelevantHeaders.put(key, value);// The header "SAP-PASSPORT" is added to metadata request even though the 'passInAllRequests' us false.
	        
		headers.put(key, value);// All headers must be considered for the actual OData operation.
		return this;

	}

	/**
	 * Builds an ODataDeleteRequest from this builder.
	 * 
	 * @return ODataDeleteRequest
	 */
	public ODataDeleteRequest build() {
		return new ODataDeleteRequestImpl(serviceName, entitySetName, keys, errorHandler, headers,
				destinationRelevantHeaders, cacheMetadata,metadataFilePath,cacheKey,isCacheRefresh);
	}
}
