/*******************************************************************************
 * (c) 201X SAP SE or an SAP affiliate company. All rights reserved.
 ******************************************************************************/
package com.sap.cloud.sdk.odatav2.connectivity;

import java.net.URL;
import java.util.HashMap;
import java.util.Map;

import com.google.common.collect.Maps;
import com.sap.cloud.sdk.cloudplatform.cache.CacheKey;
import com.sap.cloud.sdk.odatav2.connectivity.ODataQueryBuilder.ODataQueryResolver;

import lombok.EqualsAndHashCode;
import lombok.Setter;
import lombok.experimental.Accessors;

@Accessors( chain = true, fluent = true )
@EqualsAndHashCode
public class ODataReadMediaBuilder {

    private final String servicePath;
    private final String entity;
	private boolean cacheMetadata = false;
    private Map<String, String> headers = new HashMap<>();
    
    @Setter
    private boolean useMetadata = true;
    
    @Setter
    private ErrorResultHandler<?> errorHandler;    
    private final Map<String, Object> queryParams = Maps.newHashMap();

	private Map<String, Object> keys;
    private Map<String, String> destinationRelevantHeaders = new HashMap<String, String>();
    private URL metadataFilePath;
	private CacheKey cacheKey;
	private boolean isCacheRefresh = false;
        
    private ODataReadMediaBuilder(String servicePath, String entity, Map<String, Object> keys)
    {
      this.servicePath = servicePath;
      this.entity = entity;
      this.keys = keys;
    }
    
    /**
     * Adds the name of the service and entity set, containing the media resource,
     * to the ODataMediaRequestBuilder object
     * @param servicePath Name of the service including the relative path. 
     * 		  For example, in the case of OData V2, the service path is odata/v2/EPMSampleService
     * @param entitySetName Name of the entity set
     * @param keys Keys identifying the entity that contains the media resource
     * @return ODataReadMediaBuilder
     */
    public static ODataReadMediaBuilder withEntity(String servicePath, String entitySetName, Map<String, Object> keys)
    {
      return new ODataReadMediaBuilder(servicePath, entitySetName, keys);
    }
	
    /**
     * Enables caching of the metadata of an OData V2 data source. If your application is running on a tenant, then the tenant ID along with the metadata URL is used to form the cache key.
     * @return ODataReadMediaBuilder
     */
    public ODataReadMediaBuilder enableMetadataCache() {
  	  this.cacheMetadata = true;
  	return this;
    }
    
    /**
     * Enables caching of the metadata of an OData V2 data source.
     * @param key {@link com.sap.cloud.sdk.cloudplatform.cache.CacheKey Cache key} containing the ID of the tenant where the application runs. You can also include the user name in the cache key.
     * @return ODataReadMediaBuilder
     */
    public ODataReadMediaBuilder enableMetadataCache(CacheKey cacheKey){
    	this.cacheKey = cacheKey;
    	this.cacheMetadata = true;
		return this;
    }
    
     /**
	   * Replaces the existing metadata in the cache with the latest version from the OData V2 data source.
	   * @return ODataReadMediaBuilder
	   */
    public ODataReadMediaBuilder withCacheRefresh(){
    	this.isCacheRefresh = true;
    	return this;
    	
    }
    
     /**
	   * Gets the metadata from the specified path.
	   * @param metadataFilePath URL pointing to the metadata information
	   * @return ODataReadMediaBuilder A builder for forming the Create
	   */
    public ODataReadMediaBuilder withMetadata(URL metadataFilePath){
    	this.metadataFilePath = metadataFilePath;
    	return this;
    }
    
    /**
     * Adds a header to the OData request for querying media resources
     * @param key Name of the header
     * @param value Value of the header
     * @return ODataReadMediaBuilder
     */
    public ODataReadMediaBuilder withHeader(String key, String value) {
    	  return withHeader(key, value, false);
    }
    
    /**
     * Adds a header to the OData request for querying media resources
     * @param key Name of the header
     * @param value Value of the header
     * @param passInAllRequests Boolean value indicating whether the header must be passed in all the requests to the backend, 
     * 		  for example, in a $metadata call
     * @return ODataReadMediaBuilder
     */    
    public ODataReadMediaBuilder withHeader(String key, String value, boolean passInAllRequests) {
        if(passInAllRequests)
      	  destinationRelevantHeaders.put(key, value);//These headers are added to the metadata request.
        
        if(key.equals("SAP-PASSPORT") && !passInAllRequests)
            destinationRelevantHeaders.put(key, value);// The header "SAP-PASSPORT" is added to metadata request even though the 'passInAllRequests' us false.
          
        headers.put(key, value);//All headers must be considered for the actual OData operation.
        return this;
    }
        
    /**
     * Prevents the metadata being retrieved before the actual query. However, if the keys are set
     * then the metadata is fetched irrespective of whether this method was called.
     * @return ODataReadMediaBuilder
     */
    public ODataReadMediaBuilder withoutMetadata() {
        useMetadata = false;
        return this;
    }
    
    /**
     * Adds additional query parameters to the OData request
     * @param key Name of the parameter
     * @param value Value of the parameter
     * @return ODataReadMediaBuilder
     */
    public ODataReadMediaBuilder param( final String key, final Object value)
    {
    	queryParams.put(key, value);
        return this;
    }
    
    /**
     * Builds the final OData query for retrieving the media resource
     * @return ODataReadMediaRequest
     */
    public ODataReadMediaRequest build()
    {
    	if(keys != null) //If keys is set, then we need the metadata for serialization hence manually setting useMetadata to true.
    		useMetadata = true;
    	ODataQueryBuilder builder = ODataQueryBuilder.withEntity(servicePath, entity);
    	ODataQueryResolver resolver = builder.new ODataQueryResolver();
    	ODataQuery query = new ODataQuery(servicePath, entity, keys, resolver, errorHandler, headers, destinationRelevantHeaders, useMetadata, cacheMetadata, metadataFilePath,cacheKey,isCacheRefresh);
    	return new ODataReadMediaRequest(query);
    } 
}
