/*******************************************************************************
 * (c) 201X SAP SE or an SAP affiliate company. All rights reserved.
 ******************************************************************************/
package com.sap.cloud.sdk.odatav2.connectivity;

import java.net.URL;
import java.util.HashMap;
import java.util.Map;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.sap.cloud.sdk.cloudplatform.cache.CacheKey;
import com.sap.cloud.sdk.odatav2.connectivity.impl.ODataUpdateRequestImpl;
import com.sap.cloud.servicesdk.prov.jacksonutil.JacksonMapper;

public class ODataUpdateRequestBuilder {

  private String serviceName;
  private String entitySetName;
  private Map<String, Object> body = null;
  private Map<String, Object> keys;
  private ErrorResultHandler<?> errorHandler;
  private Map<String, String> headers = new HashMap<String, String>();
  private Map<String, String> destinationRelevantHeaders = new HashMap<String, String>();
  private boolean cacheMetadata = false;
  private URL metadataFilePath;
  private CacheKey cacheKey;
  private boolean isCacheRefresh = false;
  
  private ODataUpdateRequestBuilder(String serviceName, String entitySetName, Map<String, Object> keys) {
    this.serviceName = serviceName;
    this.entitySetName = entitySetName;
    this.keys = keys;
  }

  /**
   * Creates an ODataUpdateRequestBuilder with the given service and entity name.
   * @param serviceName name of the odata service where the developer wants to execute an update operation.
   * @param entitySetName name of the entity set, on which the developer wants to execute an update operation
   * @param keys keys identifying a particular entity
   * @return ODataUpdateRequestBuilder
   */
  public static ODataUpdateRequestBuilder withEntity(String serviceName, String entitySetName, Map<String, Object> keys) {
    return new ODataUpdateRequestBuilder(serviceName, entitySetName, keys);
  }

  /**
   * Adds a header to the update request.
   * @param key name of the header
   * @param value value of the header
   * @return ODataUpdateRequestBuilder
   */
  public ODataUpdateRequestBuilder withHeader(String key, String value) {
  	  return withHeader(key, value, false);
    }
    
    /**
     * Adds a header to the update request and optionally to the metadata request as well depending on the value of the 
     * passInAllRequests parameter.
     * @param key name of the header
     * @param value value of the header
     * @param passInAllRequests boolean indicating whether the header is to be passed in all the requests to the backend like $metadata call, CSRF Token fetch etc. made as part of the Update Request call.
     * @return ODataQueryBuilder
     */
    public ODataUpdateRequestBuilder withHeader(String key, String value, boolean passInAllRequests) {
    	
      if(passInAllRequests)
    	  destinationRelevantHeaders.put(key, value);//These headers are added to the metadata request.
      
      if(key.equals("SAP-PASSPORT") && !passInAllRequests)
          destinationRelevantHeaders.put(key, value);// The header "SAP-PASSPORT" is added to metadata request even though the 'passInAllRequests' us false.
        
      headers.put(key, value);//All headers must be considered for the actual OData operation.
  	  return this;
  	  
    }
  
  @Deprecated
  public ODataUpdateRequestBuilder withBody(Map<String, Object> entityData) {
	    return this.setBody(entityData);
  }
  
  /**
   * Use this API to disable metadata caching.
   * If used then metadata is fetched for every request.
   * @return ODataUpdateRequestBuilder
   */
  public ODataUpdateRequestBuilder enableMetadataCache() {
	  this.cacheMetadata = true;
	return this;
  }
  
  /**
   * Enables caching of the metadata of an OData V2 data source. If your application is running on a tenant, then the tenant ID along with the metadata URL is used to form the cache key.
   * @return ODataUpdateRequestBuilder
   */
  public ODataUpdateRequestBuilder enableMetadataCache(CacheKey cacheKey){
	  	this.cacheKey = cacheKey;
	  	this.cacheMetadata = true;
	return this;
  }
  
  /**
   * Replaces the existing metadata in the cache with the latest version from the OData V2 data source.
   * @return ODataUpdateRequestBuilder
   */
  public ODataUpdateRequestBuilder withCacheRefresh(){
  		this.isCacheRefresh = true;
  	return this;
  	
  }
  
  /**
   * Enables caching of the metadata of an OData V2 data source.
   * @param key {@link com.sap.cloud.sdk.cloudplatform.cache.CacheKey Cache key} containing the ID of the tenant where the application runs. You can also include the user name in the cache key.
   * @return ODataUpdateRequestBuilder A builder for forming the Create
   */
  public ODataUpdateRequestBuilder withMetadata(URL metadataFilePath){
  	this.metadataFilePath = metadataFilePath;
  	return this;
  }
  
  /**
   * Sets the passed map for the payload of an OData update request.
   * @param entityData
   * @return ODataCreateRequestBuilder
   */
  public ODataUpdateRequestBuilder withBodyAsMap(Map<String, Object> entityData) {
	    return this.setBody(entityData);
}
  
  /**
   * Sets an error handler to this ODataUpdateRequestBuilder.
   * @param errorHandler
   * @return ODataCreateRequestBuilder
   */
  public ODataUpdateRequestBuilder withErrorHandler(ErrorResultHandler<?> errorHandler) {
	  this.errorHandler = errorHandler;
	  return this;
  }

  private ODataUpdateRequestBuilder setBody(Map<String, Object> entityData) {
    this.body = entityData;
    return this;
  }
  
  /**
   * Builds an ODataUpdateRequest from this builder.
   * @return ODataUpdateRequest
   */
  public ODataUpdateRequest build(){
    ODataUpdateRequestImpl updateRequest = new ODataUpdateRequestImpl(
    		serviceName,
    		entitySetName,
    		keys,
    		body, 
    		errorHandler,
    		headers,
    		destinationRelevantHeaders,
    		cacheMetadata,
    		metadataFilePath,
    		cacheKey,
    		isCacheRefresh);
    return updateRequest;
  }
  
  /**
 	 * This method is used to create the request body based on a POJO object
 	 * 
 	 * @param pojoData - An instance of POJO class 
 	 * @return A ODataUpdateRequestBuilder 
 	 */

  public ODataUpdateRequestBuilder withBodyAs(Object pojoData) {
	    ObjectMapper mapper =    JacksonMapper.getMapper();
		@SuppressWarnings("unchecked")
		HashMap<String, Object> pojoInMap = mapper.convertValue(pojoData,
				HashMap.class);
		this.setBody(pojoInMap);
		return this;
}
  
}
