package com.sap.cloud.sdk.odatav2.connectivity.api;

import java.net.URL;
import java.util.List;
import java.util.Map;

import com.sap.cloud.sdk.cloudplatform.cache.CacheKey;
import com.sap.cloud.sdk.odatav2.connectivity.FilterExpression;

public interface IQueryBuilder {

	/**
	 * Selects properties to read.
	 * 
	 * @param selects The list of properties to be read.
	 * @return ODataQueryBuilder
	 */
	IQueryBuilder select(String... selects);

	/**
	 * Selects properties to read.
	 * 
	 * @param selects of type List- The list of properties to be read.
	 * @return ODataQueryBuilder
	 */
	IQueryBuilder select(List<String> selects);

	/**
	 * Enables caching of the metadata of an OData V2 data source. If your
	 * application is running on a tenant, then the tenant ID along with the
	 * metadata URL is used to form the cache key.
	 * 
	 * @return ODataQueryBuilder
	 */
	IQueryBuilder enableMetadataCache();

	/**
	 * Enables caching of the metadata of an OData V2 data source.
	 * 
	 * @param key {@link com.sap.cloud.sdk.cloudplatform.cache.CacheKey Cache key}
	 *            containing the ID of the tenant where the application runs. You
	 *            can also include the user name in the cache key.
	 * @return ODataQueryBuilder
	 */
	IQueryBuilder enableMetadataCache(CacheKey cacheKey);

	/**
	 * Replaces the existing metadata in the cache with the latest version from the
	 * OData V2 data source.
	 * 
	 * @return ODataQueryBuilder
	 */
	IQueryBuilder withCacheRefresh();

	/**
	 * Gets the metadata from the specified path.
	 * 
	 * @param metadataFilePath URL pointing to the metadata information
	 * @return ODataQueryBuilder A builder for forming the Create
	 */
	IQueryBuilder withMetadata(URL metadataFilePath);

	/**
	 * Adds a header to the query request.
	 * 
	 * @param key   name of the header
	 * @param value value of the header
	 * @return ODataQueryBuilder
	 */
	IQueryBuilder withHeader(String key, String value);

	/**
	 * Adds a header to the query request and optionally to the metadata request as
	 * well depending on the value of the passInAllRequests parameter.
	 * 
	 * @param key               name of the header
	 * @param value             value of the header
	 * @param passInAllRequests boolean indicating whether the header is to be
	 *                          passed in all the requests to the backend like
	 *                          $metadata call etc. made as part of the Query
	 *                          Request call.
	 * @return ODataQueryBuilder
	 */
	IQueryBuilder withHeader(String key, String value, boolean passInAllRequests);

	/**
	 * Sets $inlinecount=allpages as a query parameter.
	 * 
	 * @return ODataQueryBuilder
	 */
	IQueryBuilder inlineCount();

	/**
	 * Selects navigation properties to expand.
	 * 
	 * @param expands List of navigation properties to expand
	 * @return ODataQueryBuilder
	 */
	IQueryBuilder expand(String... expands);

	/**
	 * Sets top value.
	 * 
	 * @param n the top value required
	 * @return ODataQueryBuilder
	 */
	IQueryBuilder top(Integer n);

	/**
	 * Sets skip value.
	 * 
	 * @param n the skip value required
	 * @return ODataQueryBuilder
	 */
	IQueryBuilder skip(Integer n);

	/**
	 * Adds a Filter expression to the OData query.
	 * 
	 * @param filter Object of FilterExpression that represents an OData filter
	 *               expression.
	 * @return ODataQueryBuilder
	 */
	IQueryBuilder filter(FilterExpression filter);

	/**
	 * Adds a query parameter to the OData request.
	 * 
	 * @param key
	 * @param value
	 * @return ODataQueryBuilder
	 */
	IQueryBuilder param(String key, Object value);

	/**
	 * Used to tell the framework to not get the metadata before the actual query.
	 * This is not honoured if keys are set or if filter is set because the
	 * framework uses metadata for these use cases.
	 * 
	 * @return
	 */
	IQueryBuilder withoutMetadata();

	/**
	 * Builds an OData query from this builder.
	 * 
	 * @return ODataQuery
	 */
	IODataQuery build();

	/**
	 * Adds navigation Property to the URL.
	 * @param Navigation property name to be added
	 * 
	 * @return IQueryBuilder
	 */
	IQueryBuilder navigateTo(String navigation);
	
	IQueryBuilder keys(Map<String, Object> keys);

	IQueryBuilder useMetadata(boolean useMetadata);

}