package com.sap.cloud.sdk.odatav2.connectivity.filter;

import com.sap.cloud.sdk.odatav2.connectivity.ODataProperty;
import com.sap.cloud.sdk.odatav2.connectivity.ODataType;
import com.sap.cloud.sdk.odatav2.connectivity.filter.FilterFunctions.FUNCTIONS;

public class StringFunction {
	
	/**
	 * Creates concat filter function that concatenates a property of an entity with a string operand
	 * @param property An ODataProperty object representing the property of an entity
	 * @param operand The string being concatenated to the property value 
	 * @return A StringValue object representing the concat function
	 */
	public static StringValue concat(ODataProperty property, String operand) {
		return new StringValue(FUNCTIONS.CONCAT, property, ODataType.of(operand));
	}
	
	/**
	 * Creates concat filter function that concatenates the result of a filter function with a string operand.
	 * For example, concat(concat(City,", "), "Germany").
	 * @param filterFunction A StringValue object representing a filter function
	 * @param operand The string being concatenated to the filter function
	 * @return A StringValue object representing the concat function
	 */
	public static StringValue concat(StringValue filterFunction, String operand) {
		return new StringValue(FUNCTIONS.CONCAT, filterFunction, ODataType.of(operand));
	}
	/**
	 * Creates trim filter function that trims the whitespaces from the property value of an entity
	 * @param property An ODataProperty object representing the property of an entity
	 * @return A StringValue object representing the trim function
	 */
	public static StringValue trim(ODataProperty property) {
		return new StringValue(FUNCTIONS.TRIM, property, null);
	}
	/**
	 * Creates trim filter function that trims the whitespaces from the result of a filter function
	 * @param filterFunction A StringValue object representing a filter function
	 * @return A StringValue object representing the trim function
	 */
	public static StringValue trim(StringValue filterFunction) {
		return new StringValue(FUNCTIONS.TRIM, filterFunction, null);
	}
	/**
	 * Creates toupper filter function that capitalizes the property value of an entity
	 * @param property  An ODataProperty object representing the property of an entity
	 * @return A StringValue object representing the toupper function
	 */
	public static StringValue toUpper(ODataProperty property) {
		return new StringValue(FUNCTIONS.TOUPPER, property, null);
	}
	/**
	 * Creates toupper filter function that capitalizes the result of a filter function
	 * @param filterFunction A StringValue object representing a filter function
	 * @return A StringValue object representing the toupper function
	 */
	public static StringValue toUpper(StringValue filterFunction) {
		return new StringValue(FUNCTIONS.TOUPPER, filterFunction, null);
	}
	/**
	 * Creates tolower filter function that converts the property value of an entity to lower case
	 * @param property  An ODataProperty object representing the property of an entity
	 * @return A StringValue object representing the tolower function
	 */
	public static StringValue toLower(ODataProperty property) {
		return new StringValue(FUNCTIONS.TOLOWER, property, null);
	}
	/**
	 * Creates tolower filter function that converts the result of a filter function to lower case
	 * @param filterFunction A StringValue object representing a filter function
	 * @return A StringValue object representing the tolower function
	 */
	public static StringValue toLower(StringValue property) {
		return new StringValue(FUNCTIONS.TOLOWER, property, null);
	}
	/**
	 * Creates substring filter function that extracts a string of a defined length from 
	 * the property of an entity, starting from a specified position
	 * @param property  An ODataProperty object representing the property of an entity
	 * @param pos Starting position from which the string has to extracted 
	 * @param length Number of characters to be included in the string
	 * @return A StringValue object representing the substring function
	 */
	public static StringValue substring(ODataProperty property, int pos, int length) {
		return new StringValue(FUNCTIONS.SUBSTRING, property, pos, length);
	}
	/**
	 * Creates substring filter function that extracts a string of a defined length from 
	 * the result of a filter function, starting from a specified position
	 * @param filterFunction A StringValue object representing a filter function
	 * @param pos Starting position from which the string has to extracted 
	 * @param length Number of characters to be included in the string
	 * @return A StringValue object representing the substring function
	 */
	public static StringValue substring(StringValue filterFunction, int pos, int length) {
		return new StringValue(FUNCTIONS.SUBSTRING, filterFunction, pos, length);
	}
	/**
	 * Creates substring filter function that extracts a string starting from 
	 * a defined position in the property of an entity
	 * @param property  An ODataProperty object representing the property of an entity
	 * @param pos Starting position from which the string has to extracted 
	 * @return A StringValue object representing the substring function
	 */
	public static StringValue substring(ODataProperty property, int pos) {
		return new StringValue(FUNCTIONS.SUBSTRING, property, pos, null);
	}
	/**
	 * Creates substring filter function that extracts a string starting from 
	 * a defined position in the result of another filter function
	 * @param filterFunction A StringValue object representing a filter function
	 * @param pos Starting position from which the string has to extracted 
	 * @return A StringValue object representing the substring function
	 */
	public static StringValue substring(StringValue filterFunction, int pos) {
		return new StringValue(FUNCTIONS.SUBSTRING, filterFunction, pos, null);
	}
	/**
	 * Creates replace filter function that replaces a string from the
	 * property value of an entity
	 * @param property An ODataProperty object representing the property of an entity
	 * @param find String to be replaced within the property value
	 * @param replace String that replaces the {@code find} string
	 * @return A StringValue object representing the replace function
	 */
	public static StringValue replace(ODataProperty property, String find, String replace) {
		return new StringValue(FUNCTIONS.REPLACE, property, find, replace);
	}
	/**
	 * Creates replace filter function that replaces a string from the
	 * result of another filter function
	 * @param filterFunction A StringValue object representing a filter function
	 * @param find String to be replaced within the result of the filter function
	 * @param replace String that replaces the {@code find} string
	 * @return A StringValue object representing the replace function
	 */
	public static StringValue replace(StringValue filterFunction, String find, String replace) {
		return new StringValue(FUNCTIONS.REPLACE, filterFunction, find, replace);
	}
	/**
	 * Creates startswith filter function that checks whether the property value starts 
	 * with a particular string
	 * @param property An ODataProperty object representing the property of an entity
	 * @param operand Represents the starting sequence of the property value
	 * @return A StringValue object representing the startswith function
	 */
	public static BoolValue startsWith(ODataProperty property, String operand) {
		return new BoolValue(FUNCTIONS.STARTSWITH, property, ODataType.of(operand));
	}
	
	/**
	 * Creates endswith filter function that checks whether the property value ends 
	 * with a particular string
	 * @param property An ODataProperty object representing the property of an entity
	 * @param operand Represents the ending sequence of the property value
	 * @return A StringValue object representing the endswith function
	 */
	public static BoolValue endsWith(ODataProperty property, String operand) {
		return new BoolValue(FUNCTIONS.ENDSWITH, property, ODataType.of(operand));
	}
	/**
	 * Creates issubstringof filter function that checks whether a specified string is present
	 * in the property value of an entity
	 * @param property An ODataProperty object representing the property of an entity
	 * @param operand Represents the string to be searched in the property value
	 * @return A StringValue object representing the issubstringof function
	 */
	public static BoolValue isSubstringOf(ODataProperty property, String operand) {
		return new BoolValue(FUNCTIONS.SUBSTRINGOF, property, ODataType.of(operand));
	}
	/**
	 * Creates length filter function that returns the number of characters in the property value
	 * @param property An ODataProperty object representing the property of an entity
	 * @return A StringValue object representing the length function
	 */
	public static NumValue length(ODataProperty property) {
		return new NumValue(FUNCTIONS.LENGTH, property, null);
	}
	/**
	 * Creates indexof filter function that returns the position of a specified string within the property value
	 * @param property An ODataProperty object representing the property of an entity	
	 * @param operand Represents the string to be checked in the property value
	 * @return A StringValue object representing the indexof function
	 */
	public static NumValue indexOf(ODataProperty property, String operand) {
		return new NumValue(FUNCTIONS.INDEXOF,property, ODataType.of(operand));
	}
}
