/*******************************************************************************
 * (c) 201X SAP SE or an SAP affiliate company. All rights reserved.
 ******************************************************************************/
package com.sap.cloud.sdk.odatav2.connectivity.impl;

import static com.sap.cloud.sdk.odatav2.connectivity.internal.ODataConnectivityUtil.SEPARATOR_PATH;
import static com.sap.cloud.sdk.odatav2.connectivity.internal.ODataConnectivityUtil.withSeparator;

import java.io.IOException;
import java.net.URL;
import java.util.Map;
import java.util.Map.Entry;

import org.apache.http.HttpResponse;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpDelete;
import org.apache.http.client.utils.HttpClientUtils;
import org.apache.olingo.odata2.api.edm.Edm;
import org.apache.olingo.odata2.api.edm.EdmEntitySet;
import org.apache.olingo.odata2.api.edm.EdmEntityType;
import org.apache.olingo.odata2.api.edm.EdmException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.sap.cloud.sdk.cloudplatform.cache.CacheKey;
import com.sap.cloud.sdk.cloudplatform.connectivity.Destination;
import com.sap.cloud.sdk.cloudplatform.connectivity.DestinationAccessor;
import com.sap.cloud.sdk.cloudplatform.connectivity.HttpClientAccessor;
import com.sap.cloud.sdk.cloudplatform.connectivity.WithDestinationName;
import com.sap.cloud.sdk.odatav2.connectivity.ErrorResultHandler;
import com.sap.cloud.sdk.odatav2.connectivity.ODataDeleteRequest;
import com.sap.cloud.sdk.odatav2.connectivity.ODataDeleteResult;
import com.sap.cloud.sdk.odatav2.connectivity.ODataException;
import com.sap.cloud.sdk.odatav2.connectivity.ODataExceptionType;
import com.sap.cloud.sdk.odatav2.connectivity.cache.metadata.GuavaMetadataCache;
import com.sap.cloud.sdk.odatav2.connectivity.cache.metadata.MetadataCache;
import com.sap.cloud.sdk.odatav2.connectivity.internal.EdmWithCSRF;
import com.sap.cloud.sdk.odatav2.connectivity.internal.ODataConnectivityUtil;

public class ODataDeleteRequestImpl implements ODataDeleteRequest {
	private static Logger logger = LoggerFactory.getLogger(ODataDeleteRequestImpl.class);
	private static final String APPLICATION_JSON = "application/json";
	private static final String UTF_8 = "UTF-8";
	public static final String AUTHORIZATION_HEADER = "Authorization";
	private static MetadataCache metadataCache = new GuavaMetadataCache();
	/**
	 * Content Type Constants
	 */
	public static final String CONTENT_TYPE = "Content-Type";
	public static final String ACCEPT_HEADER = "Accept";
	public static final String CONTENTTYPE_ATOM_XML = "application/xml";
  private String serviceName;
  private String entitySetName;
  private Map<String, Object> keys;
	private ErrorResultHandler<?> errorHandler;
	private Map<String, String> headers;
	private Map<String, String> destinationRelevantHeaders;
	private Boolean cacheMetadata;
	private URL metadataFilePath;
	private CacheKey cacheKey;
	private Boolean isCacheRefresh;

	public ODataDeleteRequestImpl(String serviceName, String entitySetName, Map<String, Object> keys,
			ErrorResultHandler<?> errorHandler, Map<String, String> headers,
			Map<String, String> destinationRelevantHeaders, boolean metadataCache2, URL metadataFilePath,CacheKey cacheKey, boolean isCacheRefresh) {
		this.serviceName = serviceName;
		this.entitySetName = entitySetName;
		this.keys = keys;
		this.errorHandler = errorHandler;
		this.headers = headers;
		this.destinationRelevantHeaders = destinationRelevantHeaders;
		this.cacheMetadata = metadataCache2;
		this.metadataFilePath = metadataFilePath;
		this.cacheKey = cacheKey;
		this.isCacheRefresh = isCacheRefresh;
	}

	@Override
	public ODataDeleteResult execute(String destinationName) throws ODataException {
		logger.debug("Delete Called with Destination Name: " + destinationName);
		return handleExecute(destinationName, null);

	}

	private ODataDeleteResult handleExecute(String destinationName, HttpClient httpClient) throws ODataException {
		ODataDeleteResult result = null;
		httpClient = destinationName == null ? httpClient : getHttpClient(destinationName);
		if (cacheMetadata) {
			try {				
				result = delete(httpClient);
			} catch (ODataException e) {
				if (e.getODataExceptionType().equals(ODataExceptionType.OTHER)
						|| e.getODataExceptionType().equals(ODataExceptionType.ODATA_OPERATION_EXECUTION_FAILED)) {
					throw e;
				} else {
					this.isCacheRefresh = true;
					result = delete(httpClient);
				}
			}
		} else {
			result = delete(httpClient);
		}

		return result;
	}

	ODataDeleteResult delete(HttpClient httpClient) throws ODataException {
		String serviceUri = this.serviceName;
		String entitySetName = this.entitySetName;
		String completeUrl = null;
		HttpResponse httpResponse = null;
		// To get the EdmEntitySet Object
		// Get the edm object first.
		Edm edm = null;
		EdmWithCSRF edmWithCSRF = null;
		try {
			edmWithCSRF = ODataConnectivityUtil.readMetadataWithCSRF(serviceUri, httpClient, destinationRelevantHeaders,
					errorHandler, cacheMetadata,metadataFilePath,cacheKey,isCacheRefresh);
		} catch (IOException e) {
			throw new ODataException(ODataExceptionType.METADATA_FETCH_FAILED, "Metadata fetch failed!", null);
		}
		if (edmWithCSRF == null || edmWithCSRF.getEdm() == null) {
			throw new ODataException(ODataExceptionType.METADATA_FETCH_FAILED, "Metadata fetch failed!", null);
		}
		edm = edmWithCSRF.getEdm();
		EdmEntitySet entitySet = null;
		EdmEntityType entityType = null;
		String keyPredicateString = null;
		try {
			entitySet = edm.getDefaultEntityContainer().getEntitySet(entitySetName);
			if (entitySet == null)
				throw new ODataException(ODataExceptionType.INVALID_ENTITY_NAME,
						"No entity with name " + entitySetName + " in the OData service", null);
			entityType = entitySet.getEntityType();
			keyPredicateString = ODataConnectivityUtil.convertKeyValuesToString(this.keys, entityType);
		} catch (EdmException e) {
			throw new ODataException(ODataExceptionType.METADATA_PARSING_FAILED, "Error while parsing the metadata.",
					e);
		}
		completeUrl = serviceUri + "/" + entitySetName + '(' + keyPredicateString + ')';
		HttpDelete requestDelete = new HttpDelete(completeUrl);

		// Set CSRF Token Header.
		requestDelete.setHeader(ODataConnectivityUtil.CSRF_HEADER, edmWithCSRF.getCsrfToken());
		requestDelete.setHeader(CONTENT_TYPE, APPLICATION_JSON);
		requestDelete.setHeader(ACCEPT_HEADER, APPLICATION_JSON);
		for (Entry<String, String> header : headers.entrySet()) {
			requestDelete.setHeader(header.getKey(), header.getValue());
		}
		try {
			httpResponse = httpClient.execute(requestDelete);
			ODataConnectivityUtil.checkHttpStatus(httpResponse, errorHandler);
		} catch (IOException e) {
			throw new ODataException(null, e.getMessage(), e);
		}finally {
			ODataConnectivityUtil.closeQuietly(httpResponse, requestDelete);
		}

		return new ODataDeleteResult(httpResponse);
	}

	protected HttpClient getHttpClient(String destinationName) {
		Destination  dest = DestinationAccessor.tryGetDestination(destinationName).get();
		return HttpClientAccessor.getHttpClient(dest.asHttp());
	}

	@Override
	public ODataDeleteResult execute(WithDestinationName withDestinationName) throws ODataException {
		return execute(withDestinationName.getDestinationName());
	}

	@Override
	public ODataDeleteResult execute(HttpClient providedClient) throws ODataException {
		logger.debug("Delete Called with Direct URL");
		return handleExecute(null, providedClient);
	}
	
	  String getServiceName() {
	    return serviceName;
	  }

	  String getEntitySetName() {
	    return entitySetName;
	  }
	  Map<String, Object> getKeys() {
	    return keys;
	  }

	  Map<String, String> getHeaders() {
	    return headers;
	  }

	
	@Override
	public String toString() {
		return "DELETE " + withSeparator(SEPARATOR_PATH, serviceName, entitySetName) + " with key " + keys.toString();
	}

}

