/*******************************************************************************
 * (c) 201X SAP SE or an SAP affiliate company. All rights reserved.
 ******************************************************************************/
package com.sap.cloud.sdk.odatav2.connectivity.internal;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.gson.JsonSyntaxException;
import lombok.RequiredArgsConstructor;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;

import static com.sap.cloud.sdk.odatav2.connectivity.ODataGsonBuilder.newGsonBuilder;

import java.util.Map;

import com.sap.cloud.sdk.cloudplatform.logging.CloudLoggerFactory;
import com.sap.cloud.sdk.odatav2.connectivity.ErrorResultHandler;
import com.sap.cloud.sdk.odatav2.connectivity.ODataException;

@RequiredArgsConstructor
public abstract class DefaultErrorResultHandler<T extends ODataException> implements ErrorResultHandler<T>
{
    private static final Logger logger = CloudLoggerFactory.getLogger(DefaultErrorResultHandler.class);
    private static final ObjectMapper om = new ObjectMapper();

    public static final String MISSING_ERROR_PARAM = "No error field found in JSON";
    public static final String MISSING_MESSAGE_OR_VALUE_PARAM = "No message or value field found in JSON";

    protected abstract Class<T> getExceptionType();

    private T build( final String json, final Object origin, final int httpStatusCode )
    {
        final Class<T> exceptionType = getExceptionType();
        T exception;
        try {
            exception = exceptionType.newInstance();
            final Map<?, ?> r = newGsonBuilder().create().fromJson(json, Map.class);
            final Object error = r.get("error");
            if( error instanceof Map ) {
                final Object code = ((Map<?, ?>) error).get("code");
                final Object message = ((Map<?, ?>) error).get("message");
                if( message instanceof Map ) {
                    final Object messageValue = ((Map<?, ?>) message).get("value");
                    if( messageValue != null ) {
                        exception.setMessage(messageValue.toString());
                    }
                }
                if( StringUtils.isEmpty(exception.getMessage()) ) {
                    exception.setMessage(MISSING_MESSAGE_OR_VALUE_PARAM);
                }
                if( code != null ) {
                    exception.setCode(code.toString());
                }
            }
        }
        catch( final JsonSyntaxException | IllegalAccessException | InstantiationException e ) {
            try {
                exception = exceptionType.newInstance();
                if( logger.isDebugEnabled() ) {
                    logger.debug("Failed to instantiate "
                        + ODataException.class.getSimpleName()
                        + " object for detailed error message.", e);
                }
            }
            catch( final Exception e1 ) {
                exception = null;
                logger.warn("Failed to instantiate " + exceptionType.getSimpleName() + " with no arguments.", e1);
            }
        }
        if( exception != null && StringUtils.isEmpty(exception.getMessage()) ) {
            exception.setMessage(MISSING_ERROR_PARAM);
        }
        return exception;
    }

    @Override
    public T createError( final String message, final Object origin, final int httpStatusCode )
    {
        return build(message, origin, httpStatusCode);
    }

    public static DefaultErrorResultHandler<ODataException> create()
    {
        return new DefaultErrorResultHandler<ODataException>()
        {
            @Override
            protected Class<ODataException> getExceptionType()
            {
                return ODataException.class;
            }
        };
    }
}
