/*******************************************************************************
 * (c) 201X SAP SE or an SAP affiliate company. All rights reserved.
 ******************************************************************************/
package com.sap.cloud.sdk.odatav2.connectivity.internal;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.http.Header;
import org.apache.http.HttpResponse;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class ODataHttpResponseWrapper {

	HttpResponse response;
	protected int responseStatusCode = 0;
	Header[] headers = null;
	protected Map<String,List<String>> headersMap = new HashMap<String,List<String>>();
	private static final String DELIMITER = ",";
	private static final String SPACE = "\\s*";
	final static Logger log = LoggerFactory.getLogger(ODataHttpResponseWrapper.class);

  public void setResponse(HttpResponse httpResponse) {
		if(httpResponse != null)
		{
			this.responseStatusCode = httpResponse.getStatusLine().getStatusCode();
			this.headersMap = setAllHaders(httpResponse.getAllHeaders());
		}
		 
	}
	
	public int getHttpStatusCode() {
		return responseStatusCode;
	}
	
	private Map<String, List<String>> setAllHaders(Header[] headers) 
	{
		for (Header header : headers)
		{
			if(header!=null)
			{
				List<String> headerValueList = new ArrayList<>();
			    String[] arr = header.getValue().trim().split(SPACE+DELIMITER+SPACE);
				headerValueList.addAll(Arrays.asList(arr));
				headersMap.put(header.getName(),(headerValueList.isEmpty()) ? null : headerValueList);
			}
		}
		return headersMap;
	}
	
	/**
	 * Returns a collection of all the header names that the response contains.
	 * @return An {@link java.util.Enumeration<java.lang.String> Enumeration<String>} object containing header names
	 */
	public Enumeration<String> getHeaderNames() {
		Set<String> headerNames = headersMap.keySet();
		return (headerNames.size() > 0 ? Collections.enumeration(headerNames) : null);
	}
	
	/**
	 * Returns a Map of all the headers that the response contains.
	 * @return An {@link java.util.Map<java.lang.String,java.util.List<java.lang.String> } object containing headers
	 */
	public Map<String,List<String>> getHeaders() {
		return headersMap;
	}
	
	/**
	 * Returns the response header Name and List of values of the specified response header as Map<String,List<String>> object.
	 * Please note that header name is case-insensitive.
	 * @param name A <code>String</code> containing the header name
	 * @return A {@link java.util.Map<java.lang.String,java.util.List<java.lang.String> } containing header name and header value List 
	 */
	public Map<String,List<String>> getHeader(String name) {
		Map<String,List<String>> headerMap = new HashMap<String,List<String>>();
		String headerName = getHeaderName(name);
		if (log.isDebugEnabled()) {log.debug("Get first header value for name "+name+". Returning value for name "+headerName+".");}
		List<String> valueList = headersMap.get(headerName);
		headerMap.put(headerName, valueList);
		return (headerMap);
	}
	
	protected String getHeaderName(String name) {
	    for (String headerName : headersMap.keySet()) {
	      if (headerName.equalsIgnoreCase(name)) {
	        return headerName;
	      }
	    }
	    return name;
	}
}
